import type { FC } from 'react';
import type { CoreUIRootProps, LayoutProps } from 'twitch-core-ui';
import {
  ReactModalRoot,
  ScGlobalStylesStatic,
  ScGlobalStylesThemed,
  ThemeRegion,
} from 'twitch-core-ui';
import type { SetRequired } from 'type-fest';

type CoreUiSsrRootProps = Omit<LayoutProps, 'className'> &
  SetRequired<Omit<CoreUIRootProps, 'cssVars'>, 'appRootElementId'>;

// istanbul ignore next: trivial
/**
 * A SSR safe equivalent of CoreUIRoot, with the following additional opinions:
 * - Clients must use styled-components (not scss or css variables).
 * - Clients must provide appRootElementId: the div (or other block element) that fully encloses your app, so probably not the body element
 *
 * This component may also be consumed in CSR apps.
 *
 * For complete Core UI theming, <Font/> should be rendered in the consuming application's HTML head to include Core UI fonts.
 */
export const CoreUiSsrRoot: FC<CoreUiSsrRootProps> = ({
  appRootElementId,
  children,
  disableHoverCSS = false,
  theme = 'system',
  ...layoutProps
}) => {
  return (
    <>
      <ScGlobalStylesStatic />
      <ThemeRegion
        cssVars={false}
        disableHoverCSS={disableHoverCSS}
        theme={theme}
        {...layoutProps}
      >
        <ScGlobalStylesThemed />
        <ReactModalRoot
          appRootElementId={appRootElementId}
          children={children}
        />
      </ThemeRegion>
    </>
  );
};

// istanbul ignore next: trivial
CoreUiSsrRoot.displayName = 'CoreUiSsrRoot';
