import type { FC } from 'react';
import styled from 'styled-components';
import type { SVGProps } from 'twitch-core-ui';
import { SVG, SVGType } from 'twitch-core-ui';
import type { SizeProps } from '../RemSVG';
import { DEFAULT_SVG_SIZE_REM, getContainerSize } from '../RemSVG';
import type { SquareSizeProps } from '../sc-mixins';
import { squareSize } from '../sc-mixins';

export const ScContainer = styled.div<SizeProps>`
  ${getContainerSize}
  position: relative;
`;

const ScBottomOffset = styled.div<SquareSizeProps>`
  bottom: 0;
  left: 0;
  ${squareSize}
  position: absolute;

  & path {
    fill-rule: nonzero;
  }
`;

const ScTopOffset = styled.div<SquareSizeProps>`
  ${squareSize}
  position: absolute;
  right: 0;
  top: 0;
`;

type ShadowedSVGProps = Omit<SVGProps, 'height' | 'type' | 'width'> & {
  /**
   * The SVGType value of the bottom (filled) SVG, which controls the color.
   * Defaults to SVGType.Brand, which results in using the brand color (purple).
   */
  bottomType?: SVGType;
  /**
   * Height in rem units. Defaults to 2.
   */
  heightRem?: number;
  /**
   * Number between 0 and 100. Defaults to 88, or a ~1/8 size reduction.
   */
  scale?: number;
  /**
   * The SVGType value of the top (non-filled) SVG, which controls the color.
   * Defaults to undefined, which results in using the base color for the theme.
   */
  topType?: SVGType;
  /**
   * Width in rem units. Defaults to 2.
   */
  widthRem?: number;
};

// istanbul ignore next: trivial
/**
 * A component that makes a stacked SVG pair, with a "bottom" version that is
 * moved down and left with filled in paths, and a "top" version that is moved
 * up and right without extra filling. Takes all of the normal SVG props except
 * `type`, as you may optionally specify `bottomType` and `topType` separately.
 * Also takes a `scale` property to control the size of the stacked SVGs
 * relative to the size of the whole component.
 *
 * Note that this may not be able to fill all SVGs, as it is dependent on how
 * their paths are defined (and whether they have closed spaces).
 */
export const ShadowedSVG: FC<ShadowedSVGProps> = ({
  bottomType = SVGType.Brand,
  topType = undefined,
  heightRem = DEFAULT_SVG_SIZE_REM,
  widthRem = DEFAULT_SVG_SIZE_REM,
  scale = 88,
  fill,
  ...props
}) => {
  return (
    <ScContainer fillParent={fill} heightRem={heightRem} widthRem={widthRem}>
      <ScBottomOffset sizePct={scale}>
        <SVG {...props} fill type={bottomType} />
      </ScBottomOffset>
      <ScTopOffset sizePct={scale}>
        <SVG
          {...props}
          fill
          // TODO CORE-UI-TYPES: Update core-ui types to accept undefined
          // https://jira.xarth.tv/browse/EMP-4748
          type={topType as SVGType}
        />
      </ScTopOffset>
    </ScContainer>
  );
};

ShadowedSVG.displayName = 'ShadowedSVG';
