import type { CSSObject } from 'styled-components';

// All of these mixins are for treating the safe area as if its 100%/100vw of
// the screen.

/**
 * Horizontally pads out the unsafe areas
 * similar to padding-left: 0, padding-right: 0
 */
// istanbul ignore next: trivial
export function safeAreaHorizontalPaddingToRemoveInsets(): CSSObject {
  return {
    paddingLeft: 'env(safe-area-inset-left)',
    paddingRight: 'env(safe-area-inset-right)',
  };
}

/**
 * Pads out the unsafe area on the bottom edge
 * similar to padding-bottom: 0
 */
// istanbul ignore next: trivial
export function safeAreaBottomPaddingToRemoveInsets(): CSSObject {
  return {
    paddingBottom: 'env(safe-area-inset-bottom)',
  };
}

type SafeAreaVWPercentOpts = {
  vw: number;
};

/**
 * Sets width as a function of vw based on safe area only
 * similar to width: 2vw (where { vw: 2 })
 */
export function safeAreaWidthInVw({ vw }: SafeAreaVWPercentOpts): CSSObject {
  return {
    width: `${vw}vw; width: calc((100vw - env(safe-area-inset-left) - env(safe-area-inset-right)) * ${
      vw / 100
    } )`,
  };
}
/**
 * Adds padding as a function of vw based on safe area only
 * similar to padding: 2vw (where { vw: 2 })
 */
export function safeAreaPaddingInVw({ vw }: SafeAreaVWPercentOpts): CSSObject {
  return {
    padding: `${vw}vw; padding: calc((100vw - env(safe-area-inset-left) - env(safe-area-inset-right)) * ${
      vw / 100
    } )`,
  };
}

type HorizontalOverflowIntoSafeAreaOpts = {
  buffer: string;
};

/**
 * Allow content to scroll horizontally into the unsafe areas
 * (useful for shelves that should flow off the edge of the screen)
 * buffer is to allow overcoming any padding between the screen edge and initial position of the scrollable content
 * similar to margin-left: 1rem, padding-left: 1rem, margin-right: 1rem, padding-right: 1rem (if $buffer = 1rem)
 */
export function horizontalOverflowIntoSafeArea({
  buffer,
}: HorizontalOverflowIntoSafeAreaOpts): CSSObject {
  return {
    'margin-left': `calc(-1 * (env(safe-area-inset-left) + ${buffer}))`,
    'margin-right': `calc(-1 * (env(safe-area-inset-right) + ${buffer}))`,
    'overflow-x': 'scroll',
    'padding-left': `calc(env(safe-area-inset-left) + ${buffer})`,
    'padding-right': `calc(env(safe-area-inset-right) + ${buffer})`,
  };
}
