import type { FC } from 'react';
import type { FormatMessage } from 'tachyon-intl';
import { useIntl } from 'tachyon-intl';
import { exhaustedCase } from 'tachyon-utils-ts';
import {
  AlignItems,
  Background,
  BorderRadius,
  ChannelStatusIndicator,
  ChannelStatusIndicatorSize,
  ChannelStatusIndicatorStatus,
  ChannelStatusTextIndicator,
  Color,
  CoreText,
  Display,
  Layout,
  SVG,
  SVGAsset,
  TextType,
} from 'twitch-core-ui';
import { StreamType } from '../utils';

export interface StreamTypeIndicatorProps {
  isOverlay: boolean;
  isPulsing: boolean;
  streamType: StreamType | undefined;
}

interface AssetAndFill {
  asset: SVGAsset;
  fill: boolean;
}
type AssetAndFillIndicators = StreamType.Premiere | StreamType.Rerun;
type AssetAndFillMapping = Record<AssetAndFillIndicators, AssetAndFill>;

export const streamTypeAssetsAndFills: AssetAndFillMapping = {
  [StreamType.Premiere]: {
    asset: SVGAsset.VideoPremiere,
    fill: false,
  },
  [StreamType.Rerun]: {
    asset: SVGAsset.VideoRerun,
    fill: false,
  },
};

function getStreamTypeText(
  streamType: StreamType,
  formatMessage: FormatMessage,
): string {
  switch (streamType) {
    case StreamType.Live:
      return formatMessage('LIVE', 'streamTypeIndicator');
    case StreamType.Premiere:
      return formatMessage('Premiere', 'streamTypeIndicator');
    case StreamType.Rerun:
      return formatMessage('Rerun', 'streamTypeIndicator');
    case StreamType.Hosting:
      return formatMessage('HOSTING', 'streamTypeIndicator');
    default:
      return exhaustedCase(streamType, '');
  }
}

export const StreamTypeIndicator: FC<StreamTypeIndicatorProps> = ({
  isOverlay,
  isPulsing,
  streamType,
}) => {
  const { formatMessage } = useIntl();

  if (streamType === StreamType.Live) {
    return (
      <ChannelStatusTextIndicator
        label={getStreamTypeText(streamType, formatMessage)}
      />
    );
  }

  const icon =
    streamType === StreamType.Hosting ? (
      <Layout alignItems={AlignItems.Center} display={Display.Flex}>
        <ChannelStatusIndicator
          pulse={isPulsing}
          size={ChannelStatusIndicatorSize.Small}
          status={ChannelStatusIndicatorStatus.Hosting}
        />
      </Layout>
    ) : streamType === StreamType.Premiere ||
      streamType === StreamType.Rerun ? (
      <SVG {...streamTypeAssetsAndFills[streamType]} height={14} width={14} />
    ) : null;

  return (
    <Layout
      background={isOverlay ? Background.Overlay : Background.Base}
      borderRadius={BorderRadius.Small}
      color={isOverlay ? Color.Overlay : Color.Base}
      display={Display.Flex}
      padding={{ x: 0.5 }}
    >
      <Layout
        alignItems={AlignItems.Center}
        display={Display.Flex}
        margin={{ right: 0.5 }}
      >
        {icon}
      </Layout>
      {streamType && (
        <CoreText type={TextType.Span}>
          {getStreamTypeText(streamType, formatMessage)}
        </CoreText>
      )}
    </Layout>
  );
};

StreamTypeIndicator.displayName = 'StreamTypeIndicator';
