import type { CoreSrcSet } from 'twitch-core-ui';
import {
  BOX_ART_ASPECT_RATIO,
  BOX_ART_DEFAULT_URL,
  BOX_ART_IMAGE_WIDTHS,
  CHANNEL_BANNER_ASPECT_RATIO,
  CHANNEL_BANNER_IMAGE_WIDTHS,
  CHANNEL_THUMBNAIL_ASPECT_RATIO,
  CHANNEL_THUMBNAIL_DEFAULT_URL,
  CHANNEL_THUMBNAIL_IMAGE_WIDTHS,
  OVERLAY_404_IMAGE_WIDTHS,
  OVERLAY_ASPECT_RATIO,
  OVERLAY_IMAGE_WIDTHS,
  STANDARD_TEMPLATE_FRAGMENT,
} from './constants';

export * from './constants';

export interface SrcAndSrcSet {
  src: string;
  srcSet: CoreSrcSet | undefined;
}

export function buildStringFromTemplate(
  template: string,
  width: number,
  height: number,
): string {
  return template
    .replace(/{width}/g, width.toString())
    .replace(/{height}/g, height.toString());
}

export const EMPTY_SRC_AND_SRC_SET: Readonly<SrcAndSrcSet> = {
  src: '',
  srcSet: undefined,
};

/**
 * Computes an image src and srcSet for a provided image template src containing
 * "{width}" and "{height}" placeholders for consumption by a CoreImage components.
 *
 * Note:
 * Even though we're generally delegating the final element construction to
 * core-ui's CoreImage, the correct order to get proper responsive behavior
 * in all browsers is sizes > srcSet > src, and thus we mirror that here
 */
export function getSrcAndSrcSet(
  imageTemplate: string | null | undefined,
  imageWidths: number[],
  aspectRatio: number,
): SrcAndSrcSet {
  if (!imageTemplate) {
    return EMPTY_SRC_AND_SRC_SET;
  }

  return imageWidths.reduce((prev, imageWidth, index) => {
    const image = buildStringFromTemplate(
      imageTemplate,
      imageWidth,
      Math.round(imageWidth * aspectRatio),
    );
    const key = `${imageWidth}w`;

    return {
      src: index === 0 ? image : prev.src,
      srcSet: {
        ...prev.srcSet,
        [key]: image,
      },
    };
  }, {} as SrcAndSrcSet);
}

export function getSrc(
  imageTemplate: string | null | undefined,
  imageWidthPx: number,
  aspectRatio: number,
): string {
  if (!imageTemplate) {
    return EMPTY_SRC_AND_SRC_SET.src;
  }

  return buildStringFromTemplate(
    imageTemplate,
    imageWidthPx,
    Math.round(imageWidthPx * aspectRatio),
  );
}

export function getBoxArtSrcAndSrcSet(
  imageTemplate: string | null | undefined,
  opts?: { fallback: false },
): SrcAndSrcSet {
  const template =
    imageTemplate || (opts?.fallback !== false ? BOX_ART_DEFAULT_URL : null);

  return getSrcAndSrcSet(template, BOX_ART_IMAGE_WIDTHS, BOX_ART_ASPECT_RATIO);
}

export function getBoxArtSrc(
  imageTemplate: string | null | undefined,
  widthPx: number,
  opts?: { fallback: false },
): string {
  const template =
    imageTemplate || (opts?.fallback !== false ? BOX_ART_DEFAULT_URL : null);

  return getSrc(template, widthPx, BOX_ART_ASPECT_RATIO);
}

/**
 * Generates a src+srcSet object from a channel thumbnail src template
 */
export function getChannelThumbnailSrcAndSrcSet(
  imageTemplate: string | null | undefined,
  opts?: { fallback: false },
): SrcAndSrcSet {
  const template =
    imageTemplate ||
    (opts?.fallback !== false ? CHANNEL_THUMBNAIL_DEFAULT_URL : null);

  return getSrcAndSrcSet(
    template,
    CHANNEL_THUMBNAIL_IMAGE_WIDTHS,
    CHANNEL_THUMBNAIL_ASPECT_RATIO,
  );
}

/**
 * Generates a src string from a channel thumbnail src template for a given
 * width. Generally `getChannelThumbnailSrcAndSrcSet` should be used for general
 * use cases, with this reserved for special exceptions.
 */
export function getChannelThumbnailSrc(
  imageTemplate: string | null | undefined,
  widthPx: number,
  opts?: { fallback: false },
): string {
  const template =
    imageTemplate ||
    (opts?.fallback !== false ? CHANNEL_THUMBNAIL_DEFAULT_URL : null);

  return getSrc(template, widthPx, CHANNEL_THUMBNAIL_ASPECT_RATIO);
}

export function getOverlaySrcAndSrcSet(imageURL: string | null): SrcAndSrcSet {
  if (imageURL && /_404/.test(imageURL)) {
    return getSrcAndSrcSet(
      imageURL,
      OVERLAY_404_IMAGE_WIDTHS,
      OVERLAY_ASPECT_RATIO,
    );
  }

  return getSrcAndSrcSet(imageURL, OVERLAY_IMAGE_WIDTHS, OVERLAY_ASPECT_RATIO);
}

export function getOverlayImageTemplate(imageURL: string): string {
  // Find the last instance of "123x654.EXT" in the string and use the template fragment to replace it.
  // Yes, this is brittle, but the image URL for an overlay can have any image file type extension or default size.
  return imageURL.replace(
    /-\d+x\d+\.([^.]+)$/,
    `${STANDARD_TEMPLATE_FRAGMENT}.$1`,
  );
}

/**
 * The API to retrieve a channel's banner image URL does not currently support
 * returning a template string, but banner images do support this convention. Therefore,
 * we manually assemble the template string so we can consume it via our srcSet utils.
 */
export function getChannelBannerImageTemplate(imageURL: string): string {
  // Find the last instance of "." in the string and insert the template fragment before it.
  // Yes, this is brittle, but the image URL for a channel banner can have any image file type extension.
  return imageURL.replace(/\.([^.]+)$/, `${STANDARD_TEMPLATE_FRAGMENT}.$1`);
}

export function getChannelBannerSrcAndSrcSet(
  bannerImageURL: string,
): SrcAndSrcSet {
  const channelBannerImageTemplate =
    getChannelBannerImageTemplate(bannerImageURL);

  return getSrcAndSrcSet(
    channelBannerImageTemplate,
    CHANNEL_BANNER_IMAGE_WIDTHS,
    CHANNEL_BANNER_ASPECT_RATIO,
  );
}
