/* eslint-disable jest/no-large-snapshots */
import { STANDARD_TEMPLATE_FRAGMENT } from './constants';
import {
  EMPTY_SRC_AND_SRC_SET,
  buildStringFromTemplate,
  getBoxArtSrc,
  getBoxArtSrcAndSrcSet,
  getChannelBannerImageTemplate,
  getChannelBannerSrcAndSrcSet,
  getChannelThumbnailSrc,
  getChannelThumbnailSrcAndSrcSet,
  getOverlayImageTemplate,
  getOverlaySrcAndSrcSet,
  getSrc,
  getSrcAndSrcSet,
} from '.';

describe('utils/image', () => {
  const standardImageTemplate =
    'https://cdn.twitch.tv/some/cool/image-{width}x{height}.jpg';

  describe('root utility funtcions', () => {
    describe(buildStringFromTemplate, () => {
      it('replaces a width and/or height placeholder when they are provided', () => {
        expect(buildStringFromTemplate('some-string', 12, 13)).toEqual(
          'some-string',
        );

        expect(buildStringFromTemplate('some-string-{width}', 12, 13)).toEqual(
          'some-string-12',
        );

        expect(buildStringFromTemplate('some-string-{height}', 12, 13)).toEqual(
          'some-string-13',
        );

        expect(
          buildStringFromTemplate('some-string-{width}x{height}', 12, 13),
        ).toEqual('some-string-12x13');
      });
    });

    describe(getSrcAndSrcSet, () => {
      it('returns an empty object if no imageTemplate is provided', () => {
        expect(getSrcAndSrcSet(null, [], 10)).toEqual(EMPTY_SRC_AND_SRC_SET);
        expect(getSrcAndSrcSet('', [], 10)).toEqual(EMPTY_SRC_AND_SRC_SET);
      });

      it('computes a valid src and srcSet', () => {
        expect(getSrcAndSrcSet(standardImageTemplate, [100, 200], 10))
          .toMatchInlineSnapshot(`
          Object {
            "src": "https://cdn.twitch.tv/some/cool/image-100x1000.jpg",
            "srcSet": Object {
              "100w": "https://cdn.twitch.tv/some/cool/image-100x1000.jpg",
              "200w": "https://cdn.twitch.tv/some/cool/image-200x2000.jpg",
            },
          }
        `);
      });
    });

    describe(getSrc, () => {
      it('returns an empty src if no imageTemplate is provided', () => {
        expect(getSrc(null, 100, 10)).toEqual(EMPTY_SRC_AND_SRC_SET.src);
        expect(getSrc('', 100, 10)).toEqual(EMPTY_SRC_AND_SRC_SET.src);
      });

      it('computes a valid src and srcSet', () => {
        expect(getSrc(standardImageTemplate, 100, 10)).toMatchInlineSnapshot(
          `"https://cdn.twitch.tv/some/cool/image-100x1000.jpg"`,
        );
      });
    });
  });

  describe('box art functions', () => {
    describe(getBoxArtSrcAndSrcSet, () => {
      it('returns the expected set of srcSets and srcs for a given template URL', () => {
        expect(getBoxArtSrcAndSrcSet(standardImageTemplate))
          .toMatchInlineSnapshot(`
          Object {
            "src": "https://cdn.twitch.tv/some/cool/image-138x184.jpg",
            "srcSet": Object {
              "138w": "https://cdn.twitch.tv/some/cool/image-138x184.jpg",
              "208w": "https://cdn.twitch.tv/some/cool/image-208x277.jpg",
              "276w": "https://cdn.twitch.tv/some/cool/image-276x368.jpg",
              "416w": "https://cdn.twitch.tv/some/cool/image-416x555.jpg",
              "52w": "https://cdn.twitch.tv/some/cool/image-52x69.jpg",
              "624w": "https://cdn.twitch.tv/some/cool/image-624x832.jpg",
              "832w": "https://cdn.twitch.tv/some/cool/image-832x1109.jpg",
            },
          }
        `);
      });

      it('falls back to the default when given a null URL', () => {
        expect(getBoxArtSrcAndSrcSet(null)).toMatchInlineSnapshot(`
          Object {
            "src": "https://static-cdn.jtvnw.net/ttv-static/404_boxart-138x184.jpg",
            "srcSet": Object {
              "138w": "https://static-cdn.jtvnw.net/ttv-static/404_boxart-138x184.jpg",
              "208w": "https://static-cdn.jtvnw.net/ttv-static/404_boxart-208x277.jpg",
              "276w": "https://static-cdn.jtvnw.net/ttv-static/404_boxart-276x368.jpg",
              "416w": "https://static-cdn.jtvnw.net/ttv-static/404_boxart-416x555.jpg",
              "52w": "https://static-cdn.jtvnw.net/ttv-static/404_boxart-52x69.jpg",
              "624w": "https://static-cdn.jtvnw.net/ttv-static/404_boxart-624x832.jpg",
              "832w": "https://static-cdn.jtvnw.net/ttv-static/404_boxart-832x1109.jpg",
            },
          }
        `);
      });

      it('returns empty data when given a null URL and fallback is false', () => {
        expect(getBoxArtSrcAndSrcSet(null, { fallback: false })).toEqual(
          EMPTY_SRC_AND_SRC_SET,
        );
      });
    });

    describe(getBoxArtSrc, () => {
      it('returns the expected src for a given template URL', () => {
        expect(getBoxArtSrc(standardImageTemplate, 100)).toMatchInlineSnapshot(
          `"https://cdn.twitch.tv/some/cool/image-100x133.jpg"`,
        );
      });

      it('falls back to the default when given a null URL', () => {
        expect(getBoxArtSrc(null, 100)).toMatchInlineSnapshot(
          `"https://static-cdn.jtvnw.net/ttv-static/404_boxart-100x133.jpg"`,
        );
      });

      it('returns empty src when given a null URL and fallback is false', () => {
        expect(getBoxArtSrc(null, 100, { fallback: false })).toEqual(
          EMPTY_SRC_AND_SRC_SET.src,
        );
      });
    });
  });

  describe('channel thumbnail functions', () => {
    describe(getChannelThumbnailSrcAndSrcSet, () => {
      it('returns the expected set of srcSets and srcs for a given template URL', () => {
        expect(getChannelThumbnailSrcAndSrcSet(standardImageTemplate))
          .toMatchInlineSnapshot(`
          Object {
            "src": "https://cdn.twitch.tv/some/cool/image-276x155.jpg",
            "srcSet": Object {
              "1242w": "https://cdn.twitch.tv/some/cool/image-1242x699.jpg",
              "1656w": "https://cdn.twitch.tv/some/cool/image-1656x932.jpg",
              "276w": "https://cdn.twitch.tv/some/cool/image-276x155.jpg",
              "414w": "https://cdn.twitch.tv/some/cool/image-414x233.jpg",
              "552w": "https://cdn.twitch.tv/some/cool/image-552x311.jpg",
              "80w": "https://cdn.twitch.tv/some/cool/image-80x45.jpg",
              "828w": "https://cdn.twitch.tv/some/cool/image-828x466.jpg",
            },
          }
        `);
      });

      it('falls back to the default when given a null URL', () => {
        expect(getChannelThumbnailSrcAndSrcSet(null)).toMatchInlineSnapshot(`
          Object {
            "src": "https://static-cdn.jtvnw.net/ttv-static/404_preview-276x155.jpg",
            "srcSet": Object {
              "1242w": "https://static-cdn.jtvnw.net/ttv-static/404_preview-1242x699.jpg",
              "1656w": "https://static-cdn.jtvnw.net/ttv-static/404_preview-1656x932.jpg",
              "276w": "https://static-cdn.jtvnw.net/ttv-static/404_preview-276x155.jpg",
              "414w": "https://static-cdn.jtvnw.net/ttv-static/404_preview-414x233.jpg",
              "552w": "https://static-cdn.jtvnw.net/ttv-static/404_preview-552x311.jpg",
              "80w": "https://static-cdn.jtvnw.net/ttv-static/404_preview-80x45.jpg",
              "828w": "https://static-cdn.jtvnw.net/ttv-static/404_preview-828x466.jpg",
            },
          }
        `);
      });

      it('returns empty data when given a null URL and fallback is false', () => {
        expect(
          getChannelThumbnailSrcAndSrcSet(null, { fallback: false }),
        ).toEqual(EMPTY_SRC_AND_SRC_SET);
      });
    });

    describe(getChannelThumbnailSrc, () => {
      it('returns the expected src for a given template URL', () => {
        expect(
          getChannelThumbnailSrc(standardImageTemplate, 100),
        ).toMatchInlineSnapshot(
          `"https://cdn.twitch.tv/some/cool/image-100x56.jpg"`,
        );
      });

      it('falls back to the default when given a null URL', () => {
        expect(getChannelThumbnailSrc(null, 100)).toMatchInlineSnapshot(
          `"https://static-cdn.jtvnw.net/ttv-static/404_preview-100x56.jpg"`,
        );
      });

      it('returns empty src when given a null URL and fallback is false', () => {
        expect(getChannelThumbnailSrc(null, 100, { fallback: false })).toEqual(
          EMPTY_SRC_AND_SRC_SET.src,
        );
      });
    });
  });

  describe('overlay functions', () => {
    describe(getOverlayImageTemplate, () => {
      it('replaces the default size of the image with the template for JPGs', () => {
        expect(
          getOverlayImageTemplate(
            'https://cdn.twitch.tv/some/cool/image-1920x1080.jpg',
          ),
        ).toEqual(
          `https://cdn.twitch.tv/some/cool/image${STANDARD_TEMPLATE_FRAGMENT}.jpg`,
        );
      });
      it('replaces the default size of the image with the template for PNGs', () => {
        expect(
          getOverlayImageTemplate(
            'https://cdn.twitch.tv/some/noncool/image-1920x1080.PNG',
          ),
        ).toEqual(
          `https://cdn.twitch.tv/some/noncool/image${STANDARD_TEMPLATE_FRAGMENT}.PNG`,
        );
      });
    });

    describe(getOverlaySrcAndSrcSet, () => {
      it('returns the expected set of srcSets and srcs for a given template URL', () => {
        expect(getOverlaySrcAndSrcSet(standardImageTemplate))
          .toMatchInlineSnapshot(`
          Object {
            "src": "https://cdn.twitch.tv/some/cool/image-320x180.jpg",
            "srcSet": Object {
              "1280w": "https://cdn.twitch.tv/some/cool/image-1280x720.jpg",
              "320w": "https://cdn.twitch.tv/some/cool/image-320x180.jpg",
              "640w": "https://cdn.twitch.tv/some/cool/image-640x360.jpg",
              "960w": "https://cdn.twitch.tv/some/cool/image-960x540.jpg",
            },
          }
        `);
      });

      it('gives only 320 and 640 for 404 images since they are not fronted by resizing support', () => {
        expect(
          getOverlaySrcAndSrcSet(
            'https://vod-secure.twitch.tv/_404/404_processing_{width}x{height}.png',
          ),
        ).toMatchInlineSnapshot(`
          Object {
            "src": "https://vod-secure.twitch.tv/_404/404_processing_320x180.png",
            "srcSet": Object {
              "320w": "https://vod-secure.twitch.tv/_404/404_processing_320x180.png",
              "640w": "https://vod-secure.twitch.tv/_404/404_processing_640x360.png",
            },
          }
        `);
      });
    });
  });

  describe('channel banner functions', () => {
    describe(getChannelBannerImageTemplate, () => {
      it('inserts the expected template string fragment before the file extension', () => {
        expect(
          getChannelBannerImageTemplate(
            'https://cdn.twitch.tv/some/cool/image.jpg',
          ),
        ).toEqual(
          `https://cdn.twitch.tv/some/cool/image${STANDARD_TEMPLATE_FRAGMENT}.jpg`,
        );

        expect(
          getChannelBannerImageTemplate(
            'https://cdn.twitch.tv/some/cool/image.PNG',
          ),
        ).toEqual(
          `https://cdn.twitch.tv/some/cool/image${STANDARD_TEMPLATE_FRAGMENT}.PNG`,
        );

        expect(
          getChannelBannerImageTemplate(
            'https://cdn.twitch.tv/some/cool/image.jp2k',
          ),
        ).toEqual(
          `https://cdn.twitch.tv/some/cool/image${STANDARD_TEMPLATE_FRAGMENT}.jp2k`,
        );
      });
    });

    describe(getChannelBannerSrcAndSrcSet, () => {
      it('returns the expected set of srcSets and srcs for a given template URL', () => {
        expect(getChannelBannerSrcAndSrcSet('https://foo.twitch.tv/image.jpg'))
          .toMatchInlineSnapshot(`
                  Object {
                    "src": "https://foo.twitch.tv/image-320x160.jpg",
                    "srcSet": Object {
                      "1280w": "https://foo.twitch.tv/image-1280x640.jpg",
                      "320w": "https://foo.twitch.tv/image-320x160.jpg",
                      "640w": "https://foo.twitch.tv/image-640x320.jpg",
                      "960w": "https://foo.twitch.tv/image-960x480.jpg",
                    },
                  }
              `);
      });
    });
  });
});
