import type { FormatMessage } from 'tachyon-intl';

/**
 * Correctly formats viewer count for display or aria label. Optionally pass a
 * pre-formatted version of the value to customize the display.
 *
 * Note that this is subtly different from formatTotalViewCount, which is
 * suitable for indicating number of total views as opposed to number of live
 * viewers.
 *
 * @example
 * const viewerCountFormatted = formatNumberShort(props.viewerCount);
 * return (
 *    <CoreText>
 *      {formatLiveViewersCount(formatMessage, props.viewerCount, viewerCountFormatted)}
 *    </CoreText>
 * );
 */
export function formatLiveViewersCount(
  formatMessage: FormatMessage,
  viewerCount: number,
  viewerCountFormatted?: string,
): string {
  if (viewerCountFormatted) {
    return formatMessage(
      '{viewerCount, plural, one {{viewerCountFormatted} viewer} other {{viewerCountFormatted} viewers}}',
      { viewerCount, viewerCountFormatted },
      'ViewerCount',
    );
  }
  return formatMessage(
    '{viewerCount, plural, one {# viewer} other {# viewers}}',
    { viewerCount },
    'ViewerCount',
  );
}
