import type { ParsedUrlQuery } from 'querystring';
import { useRouter } from 'next/router';
import type { FC } from 'react';
import { createContext, useMemo, useRef } from 'react';
import { isBrowser } from 'tachyon-utils-stdlib';
import type { CurrentRoute } from '../parseAndCompareRoute';
import { parseAndCompareRoute } from '../parseAndCompareRoute';

export interface RouterUtilsContext {
  currentAsPath: string;
  currentPathname: string;
  currentQuery: ParsedUrlQuery;
  lastPathname: string | undefined;
  pageCount: number;
}

export const routerUtilsContext = createContext<RouterUtilsContext>({
  currentAsPath: '/',
  currentPathname: '/',
  currentQuery: {},
  lastPathname: undefined,
  /**
   * Number of pages visited in the current session, starting at 1.
   */
  pageCount: 0,
});

export interface RouterUtilsRootProps {
  /**
   * The list of query param keys that should be left in the URL. All other
   * query params will be stripped.
   */
  preservedParams?: string[];
}

/**
 * RouterUtilsRoot initializes the query param capturing system and router convenience
 * functions for a Next.js application and must be put in the React tree above all router
 * util functionality from this package.
 *
 * It controls capturing the query params from the current page navigation and
 * removing them from the url while providing them to components down-tree.
 * It takes an optional prop `preservedParams` that allows certain functional query
 * params to be left in the url, such as `t` for vod offsets. The idea for this is to
 * remove things like tt_content and tt_medium from the url to allow for clean
 * appearance and shareability.
 *
 * It also interfaces with the Next router to provide some nice convenience
 * functionality like `currentPathname`, `lastPathname`, and `pageCount`.
 */
export const RouterUtilsRoot: FC<RouterUtilsRootProps> = ({
  children,
  preservedParams,
}) => {
  const router = useRouter();
  const history = useRef<CurrentRoute[]>([]);
  let currentRoute: CurrentRoute = {
    asPath: router.asPath,
    pathname: router.pathname,
    query: router.query,
  };

  // We only strip & replace non-preserved query params on the client because:
  // 1. Our router is read-only on the server.
  // 2. Our tracking contexts are accurate at client init.
  if (isBrowser()) {
    const { changed, route: changedRoute } = parseAndCompareRoute(
      currentRoute,
      preservedParams || [],
    );

    if (changed) {
      currentRoute = changedRoute;
      router.replace(currentRoute.pathname, currentRoute.asPath, {
        shallow: true,
      });
    }
  }

  if (history.current.length === 0) {
    history.current = [currentRoute];
  } else if (history.current[0].asPath !== currentRoute.asPath) {
    history.current = [currentRoute, ...history.current];
  }

  const routeHistory = history.current;
  const route = routeHistory[0];

  const ctx = useMemo(
    () => ({
      currentAsPath: route.asPath,
      currentPathname: route.pathname,
      currentQuery: route.query,
      lastPathname:
        routeHistory.length > 1 ? routeHistory[1].pathname : undefined,
      pageCount: routeHistory.length,
    }),
    [route, routeHistory],
  );

  return <routerUtilsContext.Provider children={children} value={ctx} />;
};

RouterUtilsRoot.displayName = 'RouterUtilsRoot';
