import { useRouter } from 'next/router';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import type { MockRouter } from '../testUtils';
import { mockRouter as mockRouterFn } from '../testUtils';
import { RouterUtilsRoot, routerUtilsContext } from '.';

let mockIsBrowser = true;
jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  isBrowser: () => mockIsBrowser,
}));

jest.mock('next/router', () => ({
  ...jest.requireActual('next/router'),
  useRouter: jest.fn(),
}));
const mockUseRouter = useRouter as jest.Mock;

describe(RouterUtilsRoot, () => {
  let router: MockRouter;

  const mockReplace = jest.fn();

  // Provides ease of testing around asserting against `router.replace` usage
  function mockRouter(overrides?: Partial<MockRouter>) {
    router = mockRouterFn({
      replace: mockReplace,
      ...overrides,
    });

    mockUseRouter.mockImplementationOnce(() => router);
  }

  const setup = createShallowWrapperFactory(RouterUtilsRoot, () => ({
    preservedParams: [],
  }));

  beforeEach(() => {
    mockReplace.mockReset();
    mockUseRouter.mockReset();
    mockRouter();
  });

  describe('on the client', () => {
    beforeEach(() => {
      mockIsBrowser = true;
    });

    describe('initial render', () => {
      it("derives the initial context value from Next's Router state", () => {
        const { wrapper } = setup();
        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: router.asPath,
            currentPathname: router.pathname,
            currentQuery: router.query,
            lastPathname: undefined,
            pageCount: 1,
          },
        );
      });

      it('removes non-preserved params and replaces the route with the stripped version', () => {
        mockRouter({
          query: { baz: 'boop', boing: ['1', '2'], foo: 'bar' },
        });
        const { wrapper } = setup({ preservedParams: ['foo', 'boing'] });

        const expectedAsPath = `/?boing=1&boing=2&foo=bar`;
        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: expectedAsPath,
            currentPathname: router.pathname,
            currentQuery: router.query,
            lastPathname: undefined,
            pageCount: 1,
          },
        );

        expect(router.replace).toHaveBeenCalledTimes(1);
        expect(router.replace).toHaveBeenCalledWith(
          router.pathname,
          expectedAsPath,
          {
            shallow: true,
          },
        );

        mockRouter({
          asPath: expectedAsPath,
          pathname: router.pathname,
          query: {
            boing: ['1', '2'],
            foo: 'bar',
          },
        });
        // forces updates that update() does not (in shallow rendering)
        wrapper.setProps({});

        // Ensure that an infinite loop hasn't occurred
        expect(router.replace).toHaveBeenCalledTimes(1);
      });

      it('removes entire query string when replacing route for no preserved params', () => {
        mockRouter({
          query: { baz: 'boop', foo: 'bar' },
        });
        const { wrapper } = setup({
          preservedParams: [],
        });

        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: '/',
            currentPathname: router.pathname,
            currentQuery: router.query,
            lastPathname: undefined,
            pageCount: 1,
          },
        );

        expect(router.replace).toHaveBeenCalledTimes(1);
        expect(router.replace).toHaveBeenCalledWith(router.pathname, '/', {
          shallow: true,
        });
      });
    });

    describe('when the route changes', () => {
      it('updates route history history and page count', () => {
        const { wrapper } = setup();
        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: router.asPath,
            currentPathname: router.pathname,
            currentQuery: {},
            lastPathname: undefined,
            pageCount: 1,
          },
        );

        mockRouter({
          asPath: '/foo',
          pathname: '/foo',
          query: {},
        });
        // forces updates that update() does not (in shallow rendering)
        wrapper.setProps({});

        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: '/foo',
            currentPathname: '/foo',
            currentQuery: {},
            lastPathname: '/',
            pageCount: 2,
          },
        );

        mockRouter({
          asPath: '/foo/bar',
          pathname: '/foo/bar',
          query: {},
        });
        // forces updates that update() does not (in shallow rendering)
        wrapper.setProps({});

        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: '/foo/bar',
            currentPathname: '/foo/bar',
            currentQuery: {},
            lastPathname: '/foo',
            pageCount: 3,
          },
        );
        expect(router.replace).not.toHaveBeenCalled();
      });

      it('replaces the route for the same pathname that is updated to have non-preserved params', () => {
        mockRouter({
          asPath: '/baz',
          pathname: '/baz',
          query: { foo: 'bar' },
        });
        const { wrapper } = setup({ preservedParams: ['foo'] });

        expect(router.replace).not.toHaveBeenCalled();
        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: router.asPath,
            currentPathname: router.pathname,
            currentQuery: router.query,
            lastPathname: undefined,
            pageCount: 1,
          },
        );

        mockRouter({
          asPath: '/baz?not=preserved',
          pathname: router.pathname,
          query: { not: 'preserved' },
        });
        // forces updates that update() does not (in shallow rendering)
        wrapper.setProps({});

        expect(router.replace).toHaveBeenCalledTimes(1);
        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: '/baz',
            currentPathname: '/baz',
            currentQuery: { not: 'preserved' },
            lastPathname: '/baz',
            pageCount: 2,
          },
        );
      });

      it('does not replace the route for the same pathname updated to include preserved params', () => {
        mockRouter({ asPath: '/baz', pathname: '/baz', query: {} });
        const { wrapper } = setup({
          preservedParams: ['foo'],
        });

        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: router.asPath,
            currentPathname: router.pathname,
            currentQuery: router.query,
            lastPathname: undefined,
            pageCount: 1,
          },
        );

        mockRouter({
          asPath: '/baz?foo=preserved',
          pathname: '/baz',
          query: { foo: 'preserved' },
        });
        // forces updates that update() does not (in shallow rendering)
        wrapper.setProps({});

        expect(router.replace).not.toHaveBeenCalled();
        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: '/baz?foo=preserved',
            currentPathname: '/baz',
            currentQuery: { foo: 'preserved' },
            lastPathname: '/baz',
            pageCount: 2,
          },
        );
      });

      it('preserves the dynamic path part in a query when non-preserved params exists', () => {
        // Manually scaffold router otherwise ?foo=abc is added to asPath
        mockRouter({
          asPath: '/something/bar',
          pathname: '/[dynamic]/bar',
          query: { not: 'preserved' },
        });
        router.query['dynamic'] = 'something';
        const { wrapper } = setup();

        expect(router.replace).toHaveBeenCalledTimes(1);
        expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual(
          {
            currentAsPath: '/something/bar',
            currentPathname: router.pathname,
            currentQuery: { dynamic: 'something', not: 'preserved' },
            lastPathname: undefined,
            pageCount: 1,
          },
        );
      });
    });
  });

  describe('on the server', () => {
    beforeEach(() => {
      mockIsBrowser = false;
    });

    it('does not strip non-preserved query params or replace the route', () => {
      mockRouter({ query: { not: 'preserved' } });
      const { wrapper } = setup();

      expect(wrapper.find(routerUtilsContext.Provider).prop('value')).toEqual({
        currentAsPath: router.asPath,
        currentPathname: router.pathname,
        currentQuery: router.query,
        lastPathname: undefined,
        pageCount: 1,
      });

      expect(router.replace).not.toHaveBeenCalled();
    });
  });
});
