import type { SingletonRouter } from 'next/router';
import { queryStringForPreserved } from '../queryStringForPreserved';

export type CurrentRoute = Readonly<
  Pick<SingletonRouter, 'asPath' | 'pathname' | 'query'>
>;

/**
 * For the current router information, strips un-preserved params, and returns
 * the a new route object with updated attributes. Also returns a flag
 * indicating whether or not the route was changed.
 */
export function parseAndCompareRoute(
  route: CurrentRoute,
  preservedParams: string[],
): { changed: boolean; route: CurrentRoute } {
  const asPath = route.asPath;
  const routeParts = asPath.split('?');
  const path = routeParts[0];
  const originalQueryString = routeParts[1] || '';

  const parsedQueryString = queryStringForPreserved(
    route.query,
    preservedParams,
  );
  const queryChanged = originalQueryString !== parsedQueryString;

  const parsedAsPath = queryChanged
    ? parsedQueryString
      ? `${path}?${parsedQueryString}`
      : path
    : asPath;

  return {
    changed: queryChanged,
    route: {
      asPath: parsedAsPath,
      pathname: route.pathname,
      query: route.query,
    },
  };
}
