import type { Request, Response } from 'express';
import type next from 'next';
import type { DynamicSettings } from 'tachyon-dynamic-settings';
import type { AppEnvironment } from 'tachyon-environment';

export type NextRequestHandler = ReturnType<
  ReturnType<typeof next>['getRequestHandler']
>;

export type TachyonRequestExtension<
  ReqExt extends {} = {},
  RawCustomDynamicSettings extends {} = {},
> = Readonly<ReqExt> & {
  /**
   * Environment of the current running application instance.
   */
  readonly appEnvironment: AppEnvironment;
  /**
   * Dynamic settings from the actuator service.
   */
  readonly dynamicSettings: DynamicSettings<RawCustomDynamicSettings>;
};

export type TachyonRequest<ReqExt = {}> = Request & {
  /**
   * Tachyon-specific values attached to the request object for transport
   * between the custom server and Next render code.
   */
  tachyon: TachyonRequestExtension<ReqExt>;
};

export type TachyonResponse<ResBody = any> = Response<ResBody>;

/**
 * Specifies fields present on all Tachyon page context objects extending the more
 * generic base Next types while also allowing for app-specific extensions.
 */
type TachyonPageContextBaseExtension<ReqExt = {}> = {
  req?: TachyonRequest<ReqExt>;
  res?: TachyonResponse;
};

/**
 * For altering universal context objects
 */
export type AlteredBasePageContext<BaseNextContext, ReqExt = {}> = Omit<
  BaseNextContext,
  keyof TachyonPageContextBaseExtension<ReqExt>
> &
  TachyonPageContextBaseExtension<ReqExt>;

/**
 * For altering server-only context objects
 */
export type AlteredBasePageContextRequired<BaseNextContext, ReqExt = {}> = Omit<
  BaseNextContext,
  keyof TachyonPageContextBaseExtension<ReqExt>
> &
  Required<TachyonPageContextBaseExtension<ReqExt>>;
