import type { ParsedUrlQuery } from 'querystring';
import type {
  GetServerSideProps,
  GetServerSidePropsContext,
  NextComponentType,
  NextPage,
  NextPageContext,
} from 'next';
import type {
  GraphQLTaggedNode,
  Variables as RelayQueryVariables,
} from 'react-relay/hooks';
import type { PageviewProps } from 'tachyon-event-tracker';
import type { BuildFullCacheHeaderOpts } from 'tachyon-utils-stdlib';
import type {
  AlteredBasePageContext,
  AlteredBasePageContextRequired,
} from './common';

type TachyonPageCacheControl = Partial<BuildFullCacheHeaderOpts>;

type TachyonPageFetch<Props> = {
  /**
   * Static method used during the serverside getInitialProps GQL fetch process
   * to determine whether the data returned from the API constitutes a missing
   * object and thus for the application to set the response code to 404. The
   * page will still be responsible for rendering the not found UI itself, this
   * is used only for status code setting.
   *
   * @param queryResponse the relay query response for inspection
   * @param componentInitialProps the component's initial props
   */
  isNotFoundServerside?: (props: Props) => boolean;
  query?: GraphQLTaggedNode | undefined;
};

/**
 * Static method used to provide pageview tracking data.
 *
 * @param props the relay query response
 * @param initialProps the component's initial props
 */
export type PageviewTracking<Props> = {
  pageviewTracking: (props: Props) => PageviewProps;
};

export type TachyonPageContext<
  Query = {},
  ReqExt = {},
  PageContextExt = {},
> = AlteredBasePageContext<NextPageContext, ReqExt> &
  PageContextExt & {
    /**
     * The user's top language preference.
     */
    language: string;
    query: NextPageContext['query'] & Query;
  };

export type TachyonPageInitialProps = {
  queryVariables?: RelayQueryVariables;
};

export type TachyonPageProps = TachyonPageInitialProps;

/**
 * Type for consuming a Tachyon page within Next's App component.
 */
export type TachyonPageType<
  InitialProps = TachyonPageInitialProps,
  Props extends InitialProps = InitialProps,
  PageExtensions = {},
> = NextComponentType<TachyonPageContext, InitialProps, Props> &
  PageExtensions &
  PageviewTracking<Props> &
  TachyonPageCacheControl &
  TachyonPageFetch<Props>;

/**
 * Type for creating a Tachyon application page.
 */
export type TachyonPage<
  InitialProps = TachyonPageInitialProps,
  Props extends InitialProps = InitialProps,
  Query = {},
  ReqExt = {},
  PageContextExt = {},
  PageExtensions = {},
> = Omit<NextPage<Props, InitialProps>, 'getInitialProps'> &
  PageExtensions &
  PageviewTracking<Props> &
  TachyonPageCacheControl &
  TachyonPageFetch<Props> & {
    getInitialProps?: (
      ctx: TachyonPageContext<Query, ReqExt, PageContextExt>,
    ) => InitialProps | Promise<InitialProps>;
    // Name property gets lost without this.
    // This causes the bare use of "describe(<SomeComponent>)" to emit TS errors in test files.
    (props: Props): JSX.Element | null;
  };

/**
 * Redefinition of GetServerSidePropsContext using custom req/res
 */
type TachyonGetServerSidePropsContext<
  Query = {},
  ReqExt = {},
> = AlteredBasePageContextRequired<
  GetServerSidePropsContext<ParsedUrlQuery & Query>,
  ReqExt
>;

/**
 * Redefinition of GetServerSideProps using custom context
 */
export type TachyonGetServerSideProps<Props = {}, Query = {}, ReqExt = {}> = (
  context: TachyonGetServerSidePropsContext<Query, ReqExt>,
) => ReturnType<GetServerSideProps<Props, ParsedUrlQuery & Query>>;
