import type { FC } from 'react';
import { createContext, useContext, useState } from 'react';
import type { Notifier } from '../notifier';
import { notificationManager } from '../notifier';
import { useSetTimeoutFn } from '../useSetTimeoutFn';

export function defaultNotificationContext(): Notifier {
  return {
    dismiss: () => undefined,
    getNotifications: () => [],
    publish: (_notification) => 'fake-id',
  };
}

// istanbul ignore next: trivial
export const notificationContext = createContext<Notifier>(
  defaultNotificationContext(),
);

/**
 * Notification Context Provider.
 */
export const NotificationContextRoot: FC = ({ children }) => {
  const timeout = useSetTimeoutFn();
  const [notifier, setNotifier] = useState(
    (): Notifier =>
      // creates new object reference so that context consumers are notified of the change
      notificationManager(timeout, () => setNotifier({ ...notifier })),
  );

  return <notificationContext.Provider children={children} value={notifier} />;
};

NotificationContextRoot.displayName = 'NotificationContextRoot';

export type NotificationPublisher<Meta extends {} = {}> = Pick<
  Notifier<Meta>,
  'dismiss' | 'publish'
>;

// istanbul ignore next: trivial
/**
 * Publish and dismiss notifications.
 */
export function useNotificationPublisher<
  Meta extends {} = {},
>(): NotificationPublisher<Meta> {
  return useContext(notificationContext);
}

// istanbul ignore next: trivial
/**
 * Consume notifications.
 *
 * limit: Maximum number of notifications to consume. Defaults to all notifications.
 * order: Ordering of the messages. Either 'fifo' or 'lifo'.
 *
 */
export function useNotificationConsumer<Meta extends {} = {}>(
  ...args: Parameters<Notifier<Meta>['getNotifications']>
): ReturnType<Notifier<Meta>['getNotifications']> {
  return useContext(notificationContext).getNotifications(
    ...args,
  ) as ReturnType<Notifier<Meta>['getNotifications']>;
}
