import type { AnyFunction } from 'tachyon-type-library';
import { uniqueIDGenerator } from 'tachyon-utils-twitch';

type NotificationType = 'error' | 'info' | 'success' | 'warning';

type NotificationConsumerOpts = {
  limit?: number;
  order?: 'fifo' | 'lifo';
};

type CreateNotification<Meta extends {} = {}> = {
  /** the amount of time (ms) before auto dismissing */
  autoCloseMs?: number;
  meta: Meta;
  type: NotificationType;
};

export type Notification<Meta extends {} = {}> = CreateNotification<Meta> & {
  /** the time (in absolute ms) that the notification will dismiss */
  autoCloseAtMs: number;
  /** dismiss the notification */
  dismiss: () => void;
  /** autoincrementing. can be used to dismiss a specific notification */
  id: string;
  /**
   * starts the auto dismiss timer. this is used internally to start the
   * auto dismiss timer once a consumer has consumed the notification.
   */
  startAutoClose: (() => void) | undefined;
};

type InternalNotification<Meta extends {} = {}> = Notification<Meta> & {
  /**
   * starts the auto dismiss timer. this is used internally to start the
   * auto dismiss timer once a consumer has consumed the notification.
   */
  startAutoClose: (() => void) | undefined;
};

type Timeout = (fn: AnyFunction, delay: number) => void;

export type Notifier<Meta extends {} = {}> = {
  dismiss: (id: Notification<Meta>['id']) => void;
  getNotifications: (opts?: NotificationConsumerOpts) => Notification<Meta>[];
  publish: (notification: CreateNotification<Meta>) => Notification<Meta>['id'];
};

export function notificationManager<Meta extends {} = {}>(
  timeout: Timeout,
  onChange?: () => void,
): Notifier<Meta> {
  /*
   * We mutate this array in 'publish' and 'dismiss' because auto-dismissed timers
   * close over this array. If we reassigned the array, the timers could cause
   * a stale array reference to be reified.
   */
  const notifications: InternalNotification<Meta>[] = [];

  function notify(): void {
    onChange?.();
  }

  function dismiss(id: Notification<Meta>['id']): void {
    const idx = notifications.findIndex(
      (notification) => notification.id === id,
    );
    if (idx !== -1) {
      notifications.splice(idx, 1);
    }
    notify();
  }

  function publish(
    notification: CreateNotification<Meta>,
  ): Notification<Meta>['id'] {
    const id = uniqueIDGenerator();

    let autoCloseTimerStarted = false;
    const newNotification: InternalNotification<Meta> = {
      ...notification,
      autoCloseAtMs: Infinity,
      dismiss: () => dismiss(id),
      id,
      startAutoClose: () => {
        if (!autoCloseTimerStarted && notification.autoCloseMs) {
          autoCloseTimerStarted = true;
          newNotification.autoCloseAtMs = Date.now() + notification.autoCloseMs;
          timeout(() => dismiss(id), notification.autoCloseMs);
        }
      },
    };

    notifications.push(newNotification);
    notify();

    return id;
  }

  function getNotifications({
    limit,
    order = 'fifo',
  }: NotificationConsumerOpts = {}): Notification<Meta>[] {
    const _notifications = (
      order === 'fifo' ? notifications : notifications.reverse()
    ).slice(0, limit);

    _notifications.forEach((notification) => notification.startAutoClose?.());

    return _notifications;
  }

  return {
    dismiss,
    getNotifications,
    publish,
  };
}
