import { notificationManager } from '.';

describe(notificationManager, () => {
  describe('.publish', () => {
    it('publishes notifications', () => {
      const notifier = notificationManager(jest.fn());
      expect(notifier.getNotifications()).toHaveLength(0);

      notifier.publish({ meta: {}, type: 'info' });

      expect(notifier.getNotifications()).toHaveLength(1);
    });

    it('dismisses notification after "autoCloseMs" when set', () => {
      const notifier = notificationManager(setTimeout);
      notifier.publish({ autoCloseMs: 5000, meta: {}, type: 'info' });
      expect(notifier.getNotifications()).toHaveLength(1);

      jest.advanceTimersByTime(5000);

      expect(notifier.getNotifications()).toHaveLength(0);
    });
  });

  describe('.dismiss', () => {
    it('dismisses notifications by id', () => {
      const notifier = notificationManager(jest.fn());
      const id = notifier.publish({ meta: {}, type: 'info' });
      expect(notifier.getNotifications()).toHaveLength(1);

      notifier.dismiss(id);

      expect(notifier.getNotifications()).toHaveLength(0);
    });
  });

  describe('.getNotifications', () => {
    describe('limit', () => {
      it('defaults to all notifications', () => {
        const notifier = notificationManager(jest.fn());
        for (let i = 0; i < 10; i++) {
          notifier.publish({ meta: {}, type: 'info' });
        }

        expect(notifier.getNotifications()).toHaveLength(10);
      });

      it('limits notifications', () => {
        const notifier = notificationManager(jest.fn());
        notifier.publish({ meta: {}, type: 'info' });
        notifier.publish({ meta: {}, type: 'info' });
        notifier.publish({ meta: {}, type: 'info' });

        expect(notifier.getNotifications({ limit: 2 })).toHaveLength(2);
      });
    });

    describe('order', () => {
      it('defaults to LIFO', () => {
        const notifier = notificationManager(jest.fn());
        notifier.publish({ meta: {}, type: 'info' });
        notifier.publish({ meta: {}, type: 'error' });
        notifier.publish({ meta: {}, type: 'warning' });

        expect(notifier.getNotifications()).toEqual(
          expect.arrayContaining([
            expect.objectContaining({ meta: {}, type: 'info' }),
            expect.objectContaining({ meta: {}, type: 'error' }),
            expect.objectContaining({ meta: {}, type: 'warning' }),
          ]),
        );
      });

      it('LIFO', () => {
        const notifier = notificationManager(jest.fn());
        notifier.publish({ meta: {}, type: 'warning' });
        notifier.publish({ meta: {}, type: 'error' });
        notifier.publish({ meta: {}, type: 'info' });

        expect(notifier.getNotifications()).toEqual(
          expect.arrayContaining([
            expect.objectContaining({ meta: {}, type: 'warning' }),
            expect.objectContaining({ meta: {}, type: 'error' }),
            expect.objectContaining({ meta: {}, type: 'info' }),
          ]),
        );
      });

      it('starts the auto dismiss timer when the notification is consumed', () => {
        const mockTimer = jest.fn();
        const notifier = notificationManager(mockTimer);
        notifier.publish({ autoCloseMs: 5000, meta: {}, type: 'info' });
        expect(mockTimer).toHaveBeenCalledTimes(0);

        notifier.getNotifications();

        expect(mockTimer).toHaveBeenCalledTimes(1);
      });
    });
  });

  describe('notification', () => {
    it('is dismissable', () => {
      const notifier = notificationManager(jest.fn());
      notifier.publish({ meta: {}, type: 'info' });
      expect(notifier.getNotifications()).toHaveLength(1);

      const notification = notifier.getNotifications()[0];
      notification.dismiss();

      expect(notifier.getNotifications()).toHaveLength(0);
    });
  });

  it('invokes the "onChange" handler when notifications change', () => {
    const onChange = jest.fn();
    const notifier = notificationManager(jest.fn(), onChange);
    expect(onChange).toHaveBeenCalledTimes(0);

    notifier.publish({ meta: {}, type: 'info' });

    expect(onChange).toHaveBeenCalledTimes(1);
  });
});
