import { useRef } from 'react';
import type { AnyFunction } from 'tachyon-type-library';
import { useConstCallback, useUnmount } from 'tachyon-utils-react';
import { isBrowser } from 'tachyon-utils-stdlib';

type Timeout = (fn: AnyFunction, delay?: number) => void;

const noopServerside = (_fn: AnyFunction, _delay = 0) => undefined;

/**
 * Returns a wrapped `setTimeout` that will cancel any outstanding timers when it's
 * consuming component unmounts.
 *
 * This is useful when the consumer needs to have multiple timers outstanding.
 */

export function useSetTimeoutFn(): Timeout {
  // Disable hook linting because this is a stable if-check
  /* eslint-disable react-hooks/rules-of-hooks */
  if (!isBrowser()) {
    return noopServerside;
  }

  const ids = useRef<number[]>([]);
  const timeout = useConstCallback((fn: AnyFunction, delay = 0) => {
    const id = window.setTimeout(() => {
      ids.current = ids.current.filter((timeoutId) => timeoutId !== id);
      fn();
    }, delay);
    ids.current.push(id);
  });

  useUnmount(() => ids.current.forEach(window.clearTimeout));

  return timeout;
}
