import { logger } from 'tachyon-logger';
import type { TachyonPageContext } from 'tachyon-next-types';
import { errorGetInitialProps } from '.';

const mockLogger = logger;
jest.mock('tachyon-logger', () => ({
  logger: {
    error: jest.fn(),
  },
}));

describe(errorGetInitialProps, () => {
  const config = { package: 'page-utils' };
  const getInitialProps = (ctx: TachyonPageContext) =>
    errorGetInitialProps(config, ctx);

  it('gets the results from the status code from the response', () => {
    const res = getInitialProps({
      res: { statusCode: 99 },
    } as TachyonPageContext);

    expect(res).toEqual(
      expect.objectContaining({
        statusCode: 99,
      }),
    );
  });

  it('gets the results from the status code from the error object', () => {
    const res = getInitialProps({
      err: { statusCode: 1000 },
    } as TachyonPageContext);

    expect(res).toEqual(
      expect.objectContaining({
        statusCode: 1000,
      }),
    );
  });

  describe('when serverside', () => {
    let mockWindow: any;
    beforeEach(() => {
      mockWindow = jest.spyOn(global as any, 'window', 'get');
      mockWindow.mockImplementation(() => undefined);
    });
    afterEach(() => {
      mockWindow.mockRestore();
    });

    describe('logs', () => {
      beforeEach(() => (mockLogger.error as jest.Mock).mockReset());

      it('the error message when present', () => {
        const error = { message: 'Oh no', statusCode: 1000 };
        getInitialProps({ err: error } as TachyonPageContext);

        expect(mockLogger.error).toHaveBeenCalledWith({
          category: 'Page-utilsErrorPage',
          error,
          message: error.message,
          package: config.package,
        });
      });

      it('the default message otherwise', () => {
        getInitialProps({} as TachyonPageContext);

        expect(mockLogger.error).toHaveBeenCalledWith({
          category: 'Page-utilsErrorPage',
          error: undefined,
          message: 'No message on error page',
          package: config.package,
        });
      });
    });
  });
});
