import type { PageviewProps } from 'tachyon-event-tracker';
import type { TachyonPageContext } from 'tachyon-next-types';
import { convertToUnsafeID, isValidObject } from 'tachyon-relay';
import { safeLogin } from 'tachyon-utils-twitch';

export type ChannelPathParameters = {
  login: string;
};

export type ChannelInitialProps<QueryVariables extends {} = {}> = {
  queryVariables: ChannelPathParameters & QueryVariables;
};

type ChannelFragment = {
  id: string;
  login: string;
};

export type ChannelStreamFragment = ChannelFragment & {
  stream: {
    id: string;
  } | null;
};

export type ChannelTrackingFragment = ChannelFragment & {
  stream: {
    game?: {
      name: string;
    } | null;
    id: string;
  } | null;
};

export type PlayingChannel<
  Channel extends ChannelStreamFragment = ChannelStreamFragment,
> = Channel & {
  login: string;
  stream: NonNullable<Channel['stream']>;
};

export type HostingChannel<
  Channel extends ChannelStreamFragment = ChannelStreamFragment,
> = Channel & {
  hosting?: Channel | null | undefined;
};

// istanbul ignore next: trivial
/**
 * Canonical test for missing data that doubles as a convenience type-guard.
 */
export function channelIsFound<Channel extends ChannelFragment>(
  channel: Channel | null | undefined,
): channel is Channel {
  return isValidObject(channel);
}

// istanbul ignore next: trivial
export function isPlayingChannel<Channel extends ChannelStreamFragment>(
  channel: Channel | null | undefined,
): channel is PlayingChannel<Channel> {
  return channelIsFound(channel) && isValidObject(channel?.stream);
}

/**
 * Returns the channel to play for a Hosting Channel. A hosted channel takes precedence
 * over the channel's own stream. Service caching can cause an overlap where
 * both the channel and the hosted channel's streams appear to be live.
 */
export function getPlayingChannel<Channel extends ChannelStreamFragment>(
  channel: HostingChannel<Channel> | null,
): PlayingChannel<Channel> | null {
  if (!channel) {
    return null;
  }

  const hostedChannel = channel.hosting;
  if (hostedChannel && isPlayingChannel(hostedChannel)) {
    return hostedChannel;
  }

  if (isPlayingChannel(channel)) {
    return channel;
  }

  return null;
}

export function channelIsNotFoundServerside({
  channel,
}: {
  channel: ChannelFragment | null;
}): boolean {
  return !channelIsFound(channel);
}

type ChannelPageview = Pick<
  PageviewProps,
  'channel' | 'channelID' | 'game' | 'isLive'
>;

export type ChannelPageviewTracking = (queryResponse: {
  channel: ChannelTrackingFragment | null;
}) => ChannelPageview;

export function channelPageviewTrackingBase(
  channel: ChannelTrackingFragment | null,
  opts?: Partial<PageviewProps>,
): PageviewProps {
  if (!channel) {
    return {};
  }

  return {
    channel: channel.login,
    channelID: convertToUnsafeID(channel.id),
    isLive: isPlayingChannel(channel),
    ...opts,
  };
}

/**
 * Utility for generating the `login` query variable (and accepting others) for
 * paths that contain a user/channel login path parameter.
 */
export function channelPathGetInitialProps<
  Context extends TachyonPageContext<ChannelPathParameters>,
  QueryVariables extends {} = {},
>(
  context: Context,
  queryVars: QueryVariables = {} as QueryVariables,
): ChannelInitialProps<QueryVariables> {
  return {
    queryVariables: {
      ...queryVars,
      login: safeLogin(context.query.login),
    },
  };
}
