import type { TachyonPageContext } from 'tachyon-next-types';
import { convertToUnsafeID, garbageId, validId } from 'tachyon-relay';
import { mockChannel, mockHostedStream } from './test-utils';
import type { ChannelPathParameters } from '.';
import {
  channelIsNotFoundServerside,
  channelPageviewTrackingBase,
  channelPathGetInitialProps,
  getPlayingChannel,
} from '.';

describe(getPlayingChannel, () => {
  it('returns stream when not hosting', () => {
    const channel = mockChannel();
    expect(getPlayingChannel(channel)).toEqual(channel);
  });

  it('returns hosted stream when hosting', () => {
    const hostStream = mockHostedStream();
    expect(
      getPlayingChannel({ ...mockChannel(), hosting: hostStream }),
    ).toEqual(hostStream);
  });
});

describe(channelIsNotFoundServerside, () => {
  it('missing channel', async () => {
    const notFound = channelIsNotFoundServerside({ channel: null });

    expect(notFound).toEqual(true);
  });

  it('garbage id', async () => {
    const notFound = channelIsNotFoundServerside({
      channel: { ...mockChannel(), id: garbageId() },
    });

    expect(notFound).toEqual(true);
  });

  it('channel with valid id', async () => {
    const notFound = channelIsNotFoundServerside({
      channel: { ...mockChannel(), id: validId() },
    });

    expect(notFound).toEqual(false);
  });
});

describe(channelPageviewTrackingBase, () => {
  it('returns basic pageview data', () => {
    const channel = mockChannel();

    expect(
      channelPageviewTrackingBase(channel, { location: 'channel_home' }),
    ).toEqual({
      channel: channel.login,
      channelID: convertToUnsafeID(channel.id),
      isLive: true,
      location: 'channel_home',
    });
  });

  it('returns empty with null queryResponse', () => {
    expect(channelPageviewTrackingBase(null)).toEqual({});
  });
});

describe(channelPathGetInitialProps, () => {
  it('generates common shape', () => {
    const context = {
      query: {
        login: 'fOo',
      },
    } as TachyonPageContext<ChannelPathParameters>;
    const res = channelPathGetInitialProps(context);

    expect(res).toEqual({
      queryVariables: {
        login: 'foo',
      },
    });
  });

  it('adds additional query variables when provided', () => {
    const vars = { extra: true };
    const context = {
      query: {
        login: 'fOo',
      },
    } as TachyonPageContext<ChannelPathParameters>;
    const res = channelPathGetInitialProps(context, vars);

    expect(res).toEqual({
      queryVariables: {
        ...vars,
        login: 'foo',
      },
    });
  });
});
