import type { TachyonPageContext } from 'tachyon-next-types';
import { convertToUnsafeID } from 'tachyon-relay';
import { mockChannel } from '../utils/test-utils';
import type { ChannelVodsPathParameters } from '.';
import {
  channelVodsGetInitialProps,
  channelVodsPageviewTracking,
  getChannelVodsFilter,
  getChannelVodsShelfType,
} from '.';

describe(channelVodsPageviewTracking, () => {
  it('returns pageview data', () => {
    const channel = mockChannel();
    const queryResponse = { channel };

    expect(channelVodsPageviewTracking(queryResponse)).toEqual({
      channel: channel.login,
      channelID: convertToUnsafeID(channel.id),
      isLive: true,
      location: 'channel_vods',
    });
  });
});

describe(getChannelVodsFilter, () => {
  it('converts certain videoshelf types to videos filter', () => {
    expect(getChannelVodsFilter('LATEST_BROADCASTS')).toBe('archives');
    expect(getChannelVodsFilter('LATEST_NON_BROADCASTS')).toBe('highlights');
    expect(getChannelVodsFilter('ALL_VIDEOS')).toBe('all');
  });

  it('returns null for unknown shelf types', () => {
    expect(getChannelVodsFilter('GAME')).toBeNull();
    expect(getChannelVodsFilter('TOP_CLIPS')).toBeNull();
    expect(getChannelVodsFilter('COLLECTION')).toBeNull();
    expect(getChannelVodsFilter('LONG_VIDEOS')).toBeNull();
    expect(getChannelVodsFilter('SHORT_VIDEOS')).toBeNull();
    expect(getChannelVodsFilter()).toBeNull();
  });
});

describe(getChannelVodsShelfType, () => {
  it('converts video filter strings back to video shelf types', () => {
    expect(getChannelVodsShelfType('all')).toBe('ALL_VIDEOS');
    expect(getChannelVodsShelfType('highlights')).toBe('LATEST_NON_BROADCASTS');
    expect(getChannelVodsShelfType('archives')).toBe('LATEST_BROADCASTS');
    expect(getChannelVodsShelfType('unknownfilter')).toBeNull();
  });
});

describe(channelVodsGetInitialProps, () => {
  it('prepares props', () => {
    const res = channelVodsGetInitialProps({
      query: {
        filter: 'archives',
        login: 'aRtHaS',
      },
    } as TachyonPageContext<ChannelVodsPathParameters>);

    expect(res).toEqual({
      activeShelfType: 'LATEST_BROADCASTS',
      queryVariables: {
        login: 'arthas',
      },
    });
  });

  it('adds additional query variables when provided', () => {
    const vars = { extra: true };
    const res = channelVodsGetInitialProps(
      {
        query: {
          filter: 'archives',
          login: 'aRtHaS',
        },
      } as TachyonPageContext<ChannelVodsPathParameters>,
      vars,
    );

    expect(res).toEqual({
      activeShelfType: 'LATEST_BROADCASTS',
      queryVariables: {
        ...vars,
        login: 'arthas',
      },
    });
  });
});
