import type { PageviewProps } from 'tachyon-event-tracker';
import { EventTrackerVODType } from 'tachyon-event-tracker';
import type { TachyonPageContext } from 'tachyon-next-types';
import { convertToUnsafeID, isValidObject } from 'tachyon-relay';
import { flattenHeaderOrParam } from 'tachyon-utils-stdlib';

export type ClipPathParameters = {
  /**
   * Present only for pages using the legacy /:login/clip/:slug path.
   */
  login?: string;
  slug: string;
};

export type ClipInitialProps<QueryVariables extends {} = {}> = {
  /**
   * Populated only for pages using the legacy /:login/clip/:slug path.
   */
  login?: string | undefined;
  queryVariables: ClipPathParameters & QueryVariables;
};

type MinimalClip = {
  broadcaster: {
    id: string;
    login: string | null;
  } | null;
  game: {
    name: string;
  } | null;
  id: string;
  slug: string;
};

type ClipQueryResponse = {
  clip: MinimalClip | null;
};

export function clipGetInitialProps<
  Context extends TachyonPageContext<ClipPathParameters>,
  QueryVariables extends {} = {},
>(
  context: Context,
  queryVars: QueryVariables = {} as QueryVariables,
): ClipInitialProps<QueryVariables> {
  return {
    login: flattenHeaderOrParam(context.query.login)?.toLowerCase(),
    queryVariables: {
      ...queryVars,
      slug: flattenHeaderOrParam(context.query.slug) ?? '',
    },
  };
}

type ClipPageview = Pick<
  PageviewProps,
  'channel' | 'channelID' | 'game' | 'vodID' | 'vodType'
>;

/**
 * Canonical test for missing data that doubles as a convenience type-guard.
 */
export function clipIsFound<Clip extends MinimalClip>(
  clip: Clip | null,
  legacyChannelPathValue?: string,
): clip is Clip {
  if (!isValidObject(clip) || !clip?.slug) {
    return false;
  }

  if (
    !clip.broadcaster ||
    (legacyChannelPathValue &&
      clip.broadcaster.login !== legacyChannelPathValue)
  ) {
    return false;
  }

  return true;
}

export function clipIsNotFoundServerside({
  clip,
  queryVariables,
}: ClipInitialProps & ClipQueryResponse): boolean {
  return !clipIsFound(clip, queryVariables.login);
}

export function clipPageviewTracking({
  clip,
}: ClipQueryResponse): ClipPageview {
  return {
    channel: clip?.broadcaster?.login ?? undefined,
    channelID: convertToUnsafeID(clip?.broadcaster?.id),
    game: clip?.game?.name,
    vodID: convertToUnsafeID(clip?.id),
    vodType: EventTrackerVODType.Clip,
  };
}
