import { internet } from 'faker';
import { EventTrackerVODType } from 'tachyon-event-tracker';
import type { TachyonPageContext } from 'tachyon-next-types';
import { convertToUnsafeID, garbageId, validId } from 'tachyon-relay';
import type { ClipPathParameters } from '.';
import {
  clipGetInitialProps,
  clipIsFound,
  clipIsNotFoundServerside,
  clipPageviewTracking,
} from '.';

describe(clipGetInitialProps, () => {
  const slug = 'foo';

  it('extracts slug', () => {
    const res = clipGetInitialProps({
      query: {
        slug,
      },
    } as TachyonPageContext<ClipPathParameters>);

    expect(res).toEqual({
      queryVariables: {
        slug,
      },
    });
  });

  it('extracts login for a legacy Clip path', () => {
    const login = 'therealderekt';

    const res = clipGetInitialProps({
      query: {
        login,
        slug,
      },
    } as TachyonPageContext<ClipPathParameters>);

    expect(res).toEqual({
      login,
      queryVariables: {
        slug,
      },
    });
  });

  it('adds additional query variables when provided', () => {
    const vars = { extra: true };

    const res = clipGetInitialProps(
      {
        query: {
          slug,
        },
      } as TachyonPageContext<ClipPathParameters>,
      vars,
    );

    expect(res).toEqual({
      queryVariables: {
        ...vars,
        slug,
      },
    });
  });
});

const mockClip = {
  broadcaster: {
    id: validId(),
    login: internet.userName(),
  },
  game: {
    name: internet.userName(),
  },
  id: validId(),
  slug: internet.userName(),
};

describe(clipIsFound, () => {
  it('missing', () => {
    const found = clipIsFound(null);

    expect(found).toEqual(false);
  });

  it('garbage id', () => {
    const found = clipIsFound({ ...mockClip, id: garbageId() });

    expect(found).toEqual(false);
  });

  it('missing slug', () => {
    const found = clipIsFound({ ...mockClip, id: validId(), slug: '' });

    expect(found).toEqual(false);
  });

  it('present', () => {
    const found = clipIsFound({ ...mockClip, id: validId(), slug: 'fOo' });

    expect(found).toEqual(true);
  });

  it('legacy path broadcaster login mismatch', () => {
    const found = clipIsFound({ ...mockClip }, 'therealderekt');

    expect(found).toEqual(false);
  });

  it('legacy path broadcaster login match', () => {
    const found = clipIsFound({ ...mockClip }, mockClip.broadcaster.login);

    expect(found).toEqual(true);
  });
});

describe(clipIsNotFoundServerside, () => {
  const queryVariables = { slug: mockClip.slug };

  it('missing', () => {
    const found = clipIsNotFoundServerside({ clip: null, queryVariables });

    expect(found).toEqual(true);
  });

  it('garbage id', () => {
    const found = clipIsNotFoundServerside({
      clip: { ...mockClip, id: garbageId() },
      queryVariables,
    });

    expect(found).toEqual(true);
  });

  it('missing slug', () => {
    const found = clipIsNotFoundServerside({
      clip: { ...mockClip, id: validId(), slug: '' },
      queryVariables,
    });

    expect(found).toEqual(true);
  });

  it('present', () => {
    const found = clipIsNotFoundServerside({
      clip: { ...mockClip, id: validId(), slug: 'fOo' },
      queryVariables,
    });

    expect(found).toEqual(false);
  });

  it('legacy broadcaster login path mismatch', () => {
    const found = clipIsNotFoundServerside({
      clip: { ...mockClip, id: validId(), slug: '' },
      queryVariables: {
        ...queryVariables,
        login: 'therealderekt',
      },
    });

    expect(found).toEqual(true);
  });

  it('legacy broadcaster login path match', () => {
    const found = clipIsNotFoundServerside({
      clip: { ...mockClip },
      queryVariables: {
        ...queryVariables,
        login: mockClip.broadcaster.login,
      },
    });

    expect(found).toEqual(false);
  });
});

describe(clipPageviewTracking, () => {
  it('returns pageview data', () => {
    expect(clipPageviewTracking({ clip: mockClip })).toEqual({
      channel: mockClip.broadcaster.login,
      channelID: convertToUnsafeID(mockClip.broadcaster.id),
      game: mockClip.game.name,
      vodID: convertToUnsafeID(mockClip.id),
      vodType: EventTrackerVODType.Clip,
    });
  });
});
