import { logFlume } from 'tachyon-event-reporter';
import { logger } from 'tachyon-logger';
import { HTTPStatusCode } from 'tachyon-type-library';
import { flumeHandler } from '.';

jest.mock('tachyon-logger', () => ({ logger: { log: jest.fn() } }));
const mockLoggerLog = logger.log as jest.Mock;

jest.mock('tachyon-event-reporter', () => ({ logFlume: jest.fn() }));
const mockLogFlume = logFlume as jest.Mock;

describe(flumeHandler, () => {
  it('rejects bad hostnames', () => {
    const req = {
      body: { data: 'data' },
      hostname: 'www.nottwitch.tv',
    } as any;
    const res = {
      sendStatus: jest.fn(),
    } as any;

    flumeHandler(req, res);
    expect(mockLoggerLog).toHaveBeenCalledWith(
      expect.objectContaining({ message: 'Bad flume request' }),
    );
    expect(mockLogFlume).not.toHaveBeenCalled();
    expect(res.sendStatus).toHaveBeenCalledWith(HTTPStatusCode.BadRequest);
  });

  it('logs valid requests', () => {
    const req = {
      body: { data: 'data' },
      hostname: 'www.twitch.tv',
    } as any;
    const res = {
      sendStatus: jest.fn(),
    } as any;

    flumeHandler(req, res);
    expect(mockLoggerLog).not.toHaveBeenCalled();
    expect(mockLogFlume).toHaveBeenCalledWith(req.body.data);
    expect(res.sendStatus).toHaveBeenCalledWith(HTTPStatusCode.NoContent);
  });

  it('silently fails when req body data is missing', () => {
    mockLogFlume.mockImplementationOnce(() => {
      throw new Error('test');
    });
    const req = {
      body: {},
      hostname: 'www.twitch.tv',
    } as any;
    const res = {
      sendStatus: jest.fn(),
    } as any;

    flumeHandler(req, res);
    expect(mockLoggerLog).not.toHaveBeenCalled();
    expect(mockLogFlume).not.toHaveBeenCalled();
    expect(res.sendStatus).toHaveBeenCalledWith(HTTPStatusCode.NoContent);
  });

  it('silently fails when req body data causes an error and logs error', () => {
    mockLogFlume.mockImplementationOnce(() => {
      throw new Error('test');
    });
    const req = {
      body: { data: 'data' },
      hostname: 'www.twitch.tv',
    } as any;
    const res = {
      sendStatus: jest.fn(),
    } as any;

    flumeHandler(req, res);
    expect(mockLoggerLog).toHaveBeenCalledWith(
      expect.objectContaining({ message: 'Error handling flume payload' }),
    );
    expect(mockLogFlume).toHaveBeenCalledWith(req.body.data);
    expect(res.sendStatus).toHaveBeenCalledWith(HTTPStatusCode.NoContent);
  });
});
