import { useRef } from 'react';
import type { PageviewProps } from 'tachyon-event-tracker';
import { EventTrackerVODType } from 'tachyon-event-tracker';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import type { TachyonPageContext } from 'tachyon-next-types';
import { convertToUnsafeID, isValidObject } from 'tachyon-relay';
import { flattenHeaderOrParam } from 'tachyon-utils-stdlib';
import { Enum } from 'tachyon-utils-ts';

export type VodPathParameters = {
  videoId: string;
};

export type VodInitialProps<QueryVariables extends {} = {}> = {
  queryVariables: QueryVariables & VodPathParameters;
};

export function vodGetInitialProps<
  Context extends TachyonPageContext<VodPathParameters>,
  QueryVariables extends {} = {},
>(
  context: Context,
  queryVars: QueryVariables = {} as QueryVariables,
): VodInitialProps<QueryVariables> {
  return {
    queryVariables: {
      ...queryVars,
      videoId: context.query.videoId,
    },
  };
}

type VideoFragment = {
  id: string;
};

export type BroadcastType =
  | '%future added value' // eslint-disable-line relay/no-future-added-value
  | 'ARCHIVE'
  | 'HIGHLIGHT'
  | 'PAST_PREMIERE'
  | 'PREMIERE_UPLOAD'
  | 'UPLOAD';

type VideoOwnerStream = {
  owner: {
    id: string;
    stream: {
      id: string;
    } | null;
  } | null;
};

type VodPageviewData = {
  video: {
    broadcastType: BroadcastType | null;
    game: {
      name: string;
    } | null;
    id: string;
    owner: {
      id: string;
      login: string;
      stream: {
        id: string;
      } | null;
    } | null;
  } | null;
};

type VideoSelf = {
  viewingHistory: {
    position: number | null;
  } | null;
} | null;

/**
 * Canonical test for missing data that doubles as a convenience type-guard.
 */
export function vodIsFound<Video extends VideoFragment>(
  video: Video | null,
): video is Video {
  return isValidObject(video);
}

export function vodIsNotFoundServerside<Video extends VideoFragment>({
  video,
}: {
  video: Video | null;
}): boolean {
  return !vodIsFound(video);
}

function vodOwnerIsStreaming<Video extends VideoOwnerStream>(
  video: Video | null,
): boolean {
  return isValidObject(video?.owner?.stream);
}

type VodPageview = Pick<
  PageviewProps,
  'channel' | 'channelID' | 'game' | 'isLive' | 'vodID' | 'vodType'
>;

export function vodPageviewTracking({ video }: VodPageviewData): VodPageview {
  return {
    channel: video?.owner?.login ?? undefined,
    channelID: convertToUnsafeID(video?.owner?.id),
    game: video?.game?.name,
    isLive: vodOwnerIsStreaming(video),
    vodID: convertToUnsafeID(video?.id),
    vodType: Enum.convertValueFromExternal(
      EventTrackerVODType,
      video?.broadcastType,
    ),
  };
}

/**
 * Converts a query value corresponding to a "?t=" URL query parameter key to a
 * video offset in seconds for the general format of: `?t=1h32m12s`.
 *
 * @return video offset in seconds.
 *
 * This should go away if we abandon Pulsar Core in favor of Player Core
 */
export function getStartTimeSeconds(
  queryTime: string | undefined,
): number | undefined {
  if (!queryTime) {
    return undefined;
  }

  const parsedQueryTime = /^(?:(\d+)[h])?(?:(\d+)[m])?(?:(\d+)[s])?$/.exec(
    queryTime,
  );
  if (parsedQueryTime) {
    // Sum the total time in seconds from the components in hours/minutes/seconds format
    // We start at index 1 because the 0th element of the regex result is the full matched string contents.
    // and the individual components begin at index 1.
    const result = parsedQueryTime
      .slice(1)
      .reduce((acc, time) => acc * 60 + (parseInt(time, 10) || 0), 0);

    return result;
  }

  const parsedNum = parseInt(queryTime, 10);

  return isNaN(parsedNum) ? undefined : parsedNum;
}

export function useStartTimeFromRouter(props: {
  queryKey: string;
}): number | undefined {
  const routerUtils = useRouterUtils();
  const startFromTime = getStartTimeSeconds(
    flattenHeaderOrParam(routerUtils.currentQuery[props.queryKey]),
  );
  const hasBeenRead = useRef(false);

  // Clear start from time after first time it is read to prevent
  // subsequent playbacks of the same video from starting from that
  // same time (for restarting from the beginning after finishing watching it)
  // this pattern is not concurrent safe and also non-obvious in what it is addressing
  // https://jira.xarth.tv/browse/EMP-4332
  if (!hasBeenRead.current) {
    hasBeenRead.current = true;
    return startFromTime;
  }
}

/**
 * Calculates the start time for a vod
 *
 * @returns video offset in seconds or undefined if no start time found.
 */
export function useStartTime(queryKey: string, videoSelf?: VideoSelf): number {
  const startFromTime = useStartTimeFromRouter({
    queryKey,
  });
  const lastPlaybackPosition = videoSelf?.viewingHistory?.position;

  return startFromTime ?? lastPlaybackPosition ?? 0;
}
