import { renderHook } from '@testing-library/react-hooks';
import { datatype, internet } from 'faker';
import { mockRouterUtils, useRouterUtils } from 'tachyon-next-routing-utils';
import type { TachyonPageContext } from 'tachyon-next-types';
import { convertToUnsafeID, garbageId, validId } from 'tachyon-relay';
import type { VodPathParameters } from '.';
import {
  getStartTimeSeconds,
  useStartTime,
  useStartTimeFromRouter,
  vodGetInitialProps,
  vodIsFound,
  vodIsNotFoundServerside,
  vodPageviewTracking,
} from '.';

jest.mock('tachyon-next-routing-utils', () => ({
  ...jest.requireActual('tachyon-next-routing-utils'),
  useRouterUtils: jest.fn(),
}));

const mockUseRouterUtils = useRouterUtils as jest.Mock;

describe(vodGetInitialProps, () => {
  const videoId = datatype.uuid();

  it('extracts videoId', () => {
    const res = vodGetInitialProps({
      query: {
        videoId,
      },
    } as TachyonPageContext<VodPathParameters>);

    expect(res).toEqual({
      queryVariables: {
        videoId,
      },
    });
  });

  it('adds additional query variables when provided', () => {
    const vars = { extra: true };

    const res = vodGetInitialProps(
      {
        query: {
          videoId,
        },
      } as TachyonPageContext<VodPathParameters>,
      vars,
    );

    expect(res).toEqual({
      queryVariables: {
        ...vars,
        videoId,
      },
    });
  });
});

const mockVideo = {
  broadcastType: 'ARCHIVE',
  game: {
    name: internet.userName(),
  },
  id: validId(),
  owner: {
    id: validId(),
    login: internet.userName(),
    stream: {
      id: validId(),
    },
  },
} as const;

const mockVideoSelf = {
  viewingHistory: {
    position: 10,
  },
};

describe(vodIsNotFoundServerside, () => {
  it('missing', () => {
    const notFound = vodIsNotFoundServerside({ video: null });

    expect(notFound).toEqual(true);
  });

  it('garbage id', () => {
    const notFound = vodIsNotFoundServerside({
      video: { ...mockVideo, id: garbageId() },
    });

    expect(notFound).toEqual(true);
  });

  it('present', () => {
    const notFound = vodIsNotFoundServerside({
      video: { ...mockVideo, id: validId() },
    });

    expect(notFound).toEqual(false);
  });
});

describe(vodIsFound, () => {
  it('missing', () => {
    const found = vodIsFound(null);

    expect(found).toEqual(false);
  });

  it('garbage id', () => {
    const found = vodIsFound({ ...mockVideo, id: garbageId() });

    expect(found).toEqual(false);
  });

  it('present', () => {
    const found = vodIsFound({ ...mockVideo, id: validId() });

    expect(found).toEqual(true);
  });
});

describe(vodPageviewTracking, () => {
  it('returns pageview data', () => {
    const queryResponse = { video: mockVideo };

    expect(vodPageviewTracking(queryResponse)).toEqual({
      channel: mockVideo.owner.login,
      channelID: convertToUnsafeID(mockVideo.owner.id),
      game: mockVideo.game.name,
      isLive: true,
      vodID: convertToUnsafeID(mockVideo.id),
      vodType: 'archive',
    });
  });
});

type StartFromTimeTestCase = {
  expected: number | undefined;
  t: string | undefined;
};

describe(getStartTimeSeconds, () => {
  it.each`
    t            | expected     | message
    ${'4h02s'}   | ${14402}     | ${'converts value to seconds from query params with hours'}
    ${'1m4s'}    | ${64}        | ${'converts value to seconds from query params with minutes'}
    ${'2h1m4s'}  | ${7264}      | ${'converts value to seconds from query params with all the values'}
    ${'22'}      | ${22}        | ${'passes value from query param when conversion is not needed'}
    ${'garbage'} | ${undefined} | ${'returns undefined if the query param was invalid'}
    ${undefined} | ${undefined} | ${'returns undefined if the query param was undefined'}
  `('$message', ({ expected, t }: StartFromTimeTestCase) => {
    expect(getStartTimeSeconds(t)).toEqual(expected);
  });
});

describe(useStartTimeFromRouter, () => {
  beforeEach(() => {
    mockUseRouterUtils.mockReturnValue(mockRouterUtils());
  });

  it('always returns undefined if no "t" in route', () => {
    mockUseRouterUtils.mockReturnValue(
      mockRouterUtils({
        currentQuery: {},
      }),
    );

    const { result } = renderHook(() =>
      useStartTimeFromRouter({
        queryKey: 't',
      }),
    );

    expect(result.current).toBeUndefined();
  });

  it('returns the time once followed by undefined if "t" in route', () => {
    const startFromTime = '1h30m';
    mockUseRouterUtils.mockReturnValue(
      mockRouterUtils({
        currentQuery: { t: startFromTime },
      }),
    );

    const { rerender, result } = renderHook(() =>
      useStartTimeFromRouter({
        queryKey: 't',
      }),
    );

    expect(result.current).toEqual(5400);
    rerender();
    expect(result.current).toBeUndefined();
  });
});

describe(useStartTime, () => {
  beforeEach(() => {
    mockUseRouterUtils.mockReturnValue(mockRouterUtils());
  });

  it('returns start time from query params over the lastPlaybackPosition', () => {
    const startFromTime = '1h30m';
    mockUseRouterUtils.mockReturnValue(
      mockRouterUtils({
        currentQuery: { t: startFromTime },
      }),
    );

    const { result } = renderHook(() => useStartTime('t', mockVideoSelf));
    expect(result.current).toBe(5400);
  });
  it('returns lastPlaybackPosition if start time is missing from query params', () => {
    const { result } = renderHook(() => useStartTime('t', mockVideoSelf));
    expect(result.current).toBe(10);
  });
  it('returns undefined if no startFromTime and lastPlaybackPosition is found', () => {
    const { result } = renderHook(() =>
      useStartTime('t', {
        viewingHistory: {
          position: null,
        },
      }),
    );
    expect(result.current).toBe(0);
  });
});
