import { stringify } from 'query-string';

export const DEFAULT_LANGUAGE = 'en';

const POLYFILL_FILENAME = 'polyfill.js';
const POLYFILL_MIN_FILENAME = 'polyfill.min.js';

type PolyfillFlags = 'always' | 'gated';
type PolyfillUnknown = 'ignore' | 'polyfill';

export interface PolyfillConfig {
  endpointPrefix: string;
  features: string[];
  flags: PolyfillFlags;
  languagePrefix: string;
  locale: string;
  minify: boolean;
  unknown: PolyfillUnknown;
}

/**
 * The list of required features required for Tachyon monorepo compatibility.
 */
export function requiredFeatures(): string[] {
  return [
    'default',
    'Array.prototype.find',
    'Array.prototype.includes',
    'Intl',
    'Object.values',
    'Object.entries',
    'UserTiming',
    'fetch',
    'navigator.sendBeacon',
    'setImmediate',
  ];
}

export function defaultConfig(): PolyfillConfig {
  return {
    // retaining old v2 url for reference or rollback
    // endpointPrefix: '//polyfill.twitchsvc.net/v2/',
    endpointPrefix: '//p.twitchcdn.net/v3/',
    features: requiredFeatures(),
    flags: 'gated',
    languagePrefix: 'Intl.~locale.',
    locale: 'en',
    minify: true,
    unknown: 'polyfill',
  };
}

export function getLanguage(locale?: string): string {
  if (!locale) {
    return DEFAULT_LANGUAGE;
  }

  return locale.split('-')[0] || DEFAULT_LANGUAGE;
}

export function getFeaturesToSupport(config: PolyfillConfig): string {
  const intlPolyfill = config.languagePrefix + getLanguage(config.locale);
  const featuresToSupport = [...config.features, intlPolyfill];

  return featuresToSupport.join(',');
}

export function getPolyfillUrl(options: Partial<PolyfillConfig> = {}): string {
  const defaults = defaultConfig();
  const config: PolyfillConfig = {
    ...defaults,
    ...options,
    features: options.features
      ? [...defaults.features, ...options.features]
      : defaults.features,
  };

  const filename = config.minify ? POLYFILL_MIN_FILENAME : POLYFILL_FILENAME;
  const endpoint = `${config.endpointPrefix}${filename}?`;

  return (
    endpoint +
    stringify({
      features: getFeaturesToSupport(config),
      flags: config.flags,
      unknown: config.unknown,
    })
  );
}
