import {
  DEFAULT_LANGUAGE,
  defaultConfig,
  getLanguage,
  getPolyfillUrl,
} from '.';

describe('polyfillUrl', () => {
  describe(getLanguage, () => {
    it.each`
      input        | expected
      ${undefined} | ${DEFAULT_LANGUAGE}
      ${''}        | ${DEFAULT_LANGUAGE}
      ${'es'}      | ${'es'}
      ${'es-MX'}   | ${'es'}
    `(
      'for $input returns $expected',
      ({ expected, input }: { expected: string; input?: string }) => {
        expect(getLanguage(input)).toEqual(expected);
      },
    );
  });

  describe(getPolyfillUrl, () => {
    it('contains the features in the config base features by default', () => {
      const result = getPolyfillUrl();
      (defaultConfig().features as string[]).forEach((feature) => {
        expect(result).toEqual(expect.stringContaining(feature));
      });
    });

    it('merges custom defined features with those of the base config', () => {
      const customFeatures = ['foo', 'bar', 'baz'];

      const result = getPolyfillUrl({ features: customFeatures });
      defaultConfig().features.forEach((feature) => {
        expect(result).toEqual(expect.stringContaining(feature));
      });

      customFeatures.forEach((feature) => {
        expect(result).toEqual(expect.stringContaining(feature));
      });
    });

    it('adds internationalization for the default language if there is no locale provided', () => {
      const result = getPolyfillUrl();
      expect(result).toEqual(
        expect.stringContaining(
          defaultConfig().languagePrefix + DEFAULT_LANGUAGE,
        ),
      );
    });

    it('adds internationalization if it is needed for a particular i18n code', () => {
      const result = getPolyfillUrl({ locale: 'es-MX' });
      expect(result).toEqual(
        expect.stringContaining(`${defaultConfig().languagePrefix}es`),
      );
    });

    it('does not minify the file if the user does not want it minified', () => {
      const result = getPolyfillUrl({ minify: false });

      expect(result).toEqual(
        expect.stringContaining(
          `${defaultConfig().endpointPrefix}polyfill.js?`,
        ),
      );
    });

    it('minifies the file if requested', () => {
      const result = getPolyfillUrl({ minify: true });

      expect(result).toEqual(
        expect.stringContaining(
          `${defaultConfig().endpointPrefix}polyfill.min.js?`,
        ),
      );
    });

    it('encodes the file', () => {
      const result = getPolyfillUrl();

      expect(result).toEqual(expect.stringContaining('%2C'));
      expect(result).toEqual(expect.not.stringContaining(','));
    });
  });
});
