#!/usr/bin/env node

/* eslint-disable @typescript-eslint/no-var-requires,import/no-internal-modules */
const { existsSync, writeFileSync } = require('fs');
const { join } = require('path');
const { cwd, exit } = require('process');
const { cosmiconfig } = require('cosmiconfig');
const { getIntrospectionQuery } = require('graphql/utilities');
const { validate } = require('jest-validate');
const mkdirp = require('mkdirp');
const fetch = require('node-fetch');

// we use the gql sandbox because introspection is disabled in production
const DEFAULT_GQL_ENDPOINT = 'https://graphql.twitch.a2z.com/master/gql';

function updateGqlSchema() {
  cosmiconfig('twitch-relay')
    .search()
    .then(({ config }) => {
      validate(config, {
        exampleConfig: {
          clientId: 'twitchClientId',
          gqlEndpoint: DEFAULT_GQL_ENDPOINT,
          schema: 'path/to/schema.json',
        },
      });
      return config;
    })
    .then(({ clientId, gqlEndpoint = DEFAULT_GQL_ENDPOINT, schema }) => {
      if (!clientId) {
        throw new Error('Config must specify a clientId value.');
      }

      if (!schema) {
        throw new Error('Config must specify a schema value.');
      }

      return fetch(gqlEndpoint, {
        body: JSON.stringify({ query: getIntrospectionQuery() }),
        headers: {
          Accept: 'application/json',
          'Client-Id': clientId,
          'Content-Type': 'application/json',
        },
        method: 'POST',
        timeout: 5000,
      })
        .then((res) => res.json())
        .then((schemaObject) => {
          // ensure the parent directory exists before trying to write
          if (schema.match(/\//)) {
            const schemaDirectory = schema.split('/').slice(0, -1).join('/');
            if (!existsSync(schemaDirectory)) {
              mkdirp.sync(schemaDirectory);
            }
          }

          writeFileSync(schema, JSON.stringify(schemaObject));
        });
    })
    .catch((err) => {
      // eslint-disable-next-line no-console
      console.error(
        'An error occured fetching the schema (you must be on the Amazon VPN):\n',
        err.message,
      );
      exit(1);
    });
}

if (!module.parent) {
  updateGqlSchema();
}

module.exports = updateGqlSchema;
