import { Component } from 'react';
import {
  QueryRenderer as OgQueryRenderer,
  ReactRelayContext,
} from 'react-relay/legacy';
import { createOperationDescriptor, getRequest } from 'relay-runtime';

export type QueryRendererProps = ConstructorParameters<
  typeof OgQueryRenderer
>[0];

export type QueryRendererRenderProps = Parameters<
  QueryRendererProps['render']
>[0];

// istanbul ignore next: would mostly be testing mocks
/**
 * This is a stripped down SSR version of the base QueryRenderer code. It merely
 * streamlines the same codepath from the original QueryRenderer when on the
 * server, forcing a store-only lookup path to avoid the Observable-based
 * network lookup logic and ensure that `queryFetcher.dispose` is called to
 * dispose of any outstanding objects. This is only a class component because
 * QueryRenderer is a class component and this mimics its API.
 */
export class TachyonQueryRenderer extends Component<QueryRendererProps> {
  public static displayName = 'TachyonQueryRenderer';

  // definitively defined for the contexts in which it is used
  snapshot!: { data: unknown };

  constructor(props: QueryRendererProps) {
    super(props);

    // we only worry about doing this in the constructor because it only
    // is necessary on the server and there are no updates to render there
    if (typeof window === 'undefined' && this.props.query) {
      const request = getRequest(this.props.query);
      const operation = createOperationDescriptor(
        request,
        this.props.variables,
      );
      this.snapshot = this.props.environment.lookup(operation.fragment);
    }
  }

  public override render(): JSX.Element {
    if (typeof window === 'undefined') {
      const { environment, query, render } = this.props;
      if (query) {
        /* eslint-disable react/jsx-no-constructed-context-values */
        return (
          <ReactRelayContext.Provider value={{ environment }}>
            {render({ error: null, props: this.snapshot.data, retry: null })}
          </ReactRelayContext.Provider>
        );
      }
      // OgQueryRenderer will happily pass through if it doesn't receive a query
      // but we ensure avoiding any side effects this way
      return <>{render({ error: null, props: {}, retry: null })}</>;
    }

    return <OgQueryRenderer {...this.props} />;
  }
}
