import { createContext, useMemo, useState } from 'react';
import { fetchQuery, graphql, useRelayEnvironment } from 'react-relay/hooks';
import { logger } from 'tachyon-logger';
import { useEffectOnce } from 'tachyon-utils-react';
import { errorMessageFromCatch } from 'tachyon-utils-stdlib';
import type { FCWithQuery } from '../types';
import type {
  RequestInfoRoot_Query,
  RequestInfoRoot_QueryResponse,
  RequestInfoRoot_QueryVariables,
} from './__generated__/RequestInfoRoot_Query.graphql';

export type RequestInfo = RequestInfoRoot_QueryResponse['requestInfo'] | null;

export interface RequestInfoContext {
  requestInfo: RequestInfo;
}

export const requestInfoContext = createContext<RequestInfoContext>({
  requestInfo: null,
});

export const RequestInfoRoot: FCWithQuery<{}, RequestInfoRoot_QueryVariables> =
  ({ children }) => {
    const relayEnvironment = useRelayEnvironment();
    const [requestInfo, setRequestInfo] = useState<RequestInfo | null>(null);
    const requestInfoContextValue = useMemo(
      () => ({ requestInfo }),
      [requestInfo],
    );

    useEffectOnce(() => {
      updateRequestInfo();
    });

    async function updateRequestInfo() {
      try {
        const res = await fetchQuery<RequestInfoRoot_Query>(
          relayEnvironment,
          RequestInfoRoot.query,
          RequestInfoRoot.variables,
        ).toPromise();

        if (!res?.requestInfo) {
          throw new Error('response missing valid data');
        }

        setRequestInfo(res.requestInfo);
      } catch (e) {
        logger.warn({
          category: 'updateRequestInfo',
          message: errorMessageFromCatch(e),
          package: 'tachyon-relay',
        });
      }
    }

    return (
      <requestInfoContext.Provider value={requestInfoContextValue}>
        {children}
      </requestInfoContext.Provider>
    );
  };

RequestInfoRoot.displayName = 'RequestInfoRoot';
RequestInfoRoot.variables = {};

RequestInfoRoot.query = graphql`
  query RequestInfoRoot_Query {
    requestInfo {
      countryCode
      fromEU
      ipAddress
      isFromEEA
    }
  }
`;
