import { logger } from 'tachyon-logger';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { RequestInfoRoot } from './RequestInfoRoot';
import type { RequestInfoProps } from './withRequestInfo';
import { withRequestInfo } from './withRequestInfo';

let mockFetchQueryValue: Promise<any> | undefined = undefined;
jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  fetchQuery: jest.fn(() => ({ toPromise: () => mockFetchQueryValue })),
  useRelayEnvironment: jest.fn(() => ({})),
}));

jest.mock('tachyon-logger', () => ({ logger: { warn: jest.fn() } }));

describe('RequestInfo', () => {
  beforeEach(() => {
    mockFetchQueryValue = undefined;
    (logger.warn as jest.Mock).mockReset();
  });

  const setup = createMountWrapperFactory(RequestInfoRoot);

  it('receives a successful response and propagates a hydrated requestInfo onject', async () => {
    const data = { requestInfo: { someRequestInfoData: 'therealderekt' } };
    mockFetchQueryValue = Promise.resolve(data);

    const propRecorder = jest.fn();
    const TestComponent = withRequestInfo((props: RequestInfoProps) => {
      propRecorder(props);
      return null;
    });

    setup({ children: <TestComponent /> });

    return Promise.resolve(() => {
      expect(propRecorder).toHaveBeenCalledWith(data);
    });
  });

  it('logs warning and does not update when response does not have request info', async () => {
    const propRecorder = jest.fn();
    const TestComponent = withRequestInfo((props: RequestInfoProps) => {
      propRecorder(props);
      return null;
    });

    setup({ children: <TestComponent /> });

    return Promise.resolve(() => {
      expect(propRecorder).toHaveBeenCalledWith({ requestInfo: null });
      expect(logger.warn).toHaveBeenCalledWith(
        expect.stringContaining('Error retrieving request info'),
      );
    });
  });

  it('logs warning and does not update when fetch errors', async () => {
    mockFetchQueryValue = Promise.reject(new Error());

    const propRecorder = jest.fn();
    const TestComponent = withRequestInfo((props: RequestInfoProps) => {
      propRecorder(props);
      return null;
    });

    setup({ children: <TestComponent /> });

    return Promise.resolve(() => {
      expect(propRecorder).toHaveBeenCalledWith({ requestInfo: null });
      expect(logger.warn).toHaveBeenCalledWith(
        expect.stringContaining('Error retrieving request info'),
      );
    });
  });
});
