import type { BackendService } from './backendServices';
import type { TwitchPayloadError } from './types';

export type ConfigureTachyonRelayOpts = {
  /**
   * Should the client retry failed fetches in the browser (default: true)
   */
  browserRetry?: boolean;
  /**
   * The Twitch API `Client-Id` value of the consuming app (required)
   */
  clientId: string;
  /**
   * Causes the error handling to console.debug more detailed error info. Ensure
   * to enable any console filters in order to see the output.
   *
   * Also attaches window.getTachyonRelayStore for visibility into the relay
   * store while debugging.
   */
  debug?: boolean;
  /**
   * Callback for handling the appearance of an errors array. If no function
   * is passed, then any errors will be treated as failed fetches in line with
   * the current guidance from the API team. This function will receive a copy
   * of the errors array and the GraphQL query string with which to determine
   * whether the application can proceed with the errors as listed.
   *
   * @param errors GraphQL errors array
   * @param query Raw GraphQL query
   * @returns boolean - true for fatal errors, false for manageable ones
   */
  errorsArrayIsFatal?: (
    errors: TwitchPayloadError[],
    queryName: string,
  ) => boolean;
  /**
   * Backend services that you want to simulate failing. Be sure to pass a
   * a fail-compatible GraphQL endpoint for this to have any effect. Requires
   * debug mode to be active.
   */
  failServices?: BackendService[];
  /**
   * The Twitch API `Client-Id` value of the consuming app
   * (default: `https://gql.twitch.tv/gql`)
   */
  gqlEndpoint?: string;
  /**
   * Timeout limit (ms) for fetches on the server (default: 1500).
   */
  serverFetchTimeout?: number;
  /**
   * Optional https agent for configuring fetch behavior on the server:
   * https://nodejs.org/api/http.html#http_new_agent_options
   * Intentionally any to avoid forcing a @types/node dependency
   */
  serverHttpsAgent?: any;
};

export const defaultConfiguration: Required<ConfigureTachyonRelayOpts> = {
  browserRetry: true,
  clientId: '',
  debug: false,
  errorsArrayIsFatal: () => true,
  failServices: [],
  gqlEndpoint: 'https://gql.twitch.tv/gql',
  serverFetchTimeout: 1500,
  serverHttpsAgent: undefined,
};

export let configuration = { ...defaultConfiguration };

// istanbul ignore next: trivial
export function configureTachyonRelay(opts: ConfigureTachyonRelayOpts): void {
  configuration = {
    ...defaultConfiguration,
    ...opts,
  };
}

// istanbul ignore next: trivial
export function resetConfiguration(): void {
  configuration = { ...defaultConfiguration };
}
