import type { GraphQLNode, GraphQLResponseNode } from '../types';

// the separator and regexes need to stay in sync
export const GQL_SAFE_ID_SEPARATOR = '}|{';
// non-global because it works on 1 id at a time
export const GQL_SAFE_ID_REGEX = /[\w\d]+\}\|\{([\w\d]+)/;

export const GQL_SAFE_EVENT_TYPE = 'Event';
// the id given to objects that the API _should've_ returned as null
export const GQL_GARBAGE_OBJECT_ID = 'GarbageObject';

// The purpose of these utilities is to create an environment where Relay is
// able to work with (synthetic) globally unique IDs for all nodes/objects.
// This is a requirement for Relay Modern, and because our API doesn't properly
// support this yet, we can use these functions at the edges of the app (e.g.
// when processing API responses or interacting with the current url/location)
// to convert between "safe" (unique) and "unsafe" (raw) id values as needed.

/**
 * Adds an accompanying __typename to all id fields in a graphql request
 *
 * @param query the string representing a graphql query
 * @returns the same query with __typename added to all nodes that request id
 */
export function convertGqlQueryToIncludeTypename(query: string): string {
  return query.replace(/\n( +)id/g, '\n$1id\n$1__typename');
}

/**
 * Converts all values in an object from the safe-id pattern to their raw
 * underlying unsafe ids. Works recursively through objects as well.
 *
 * @param obj object potentially containing safe ids
 * @returns the converted object with unsafe ids
 */
export function convertGqlIdValuesToUnsafe<T extends Record<any, any>>(
  obj: T,
): T {
  return Object.entries(obj).reduce(
    (acc, [key, value]: [keyof T, T[keyof T]]) => {
      if (Array.isArray(value)) {
        acc[key] = value.map(convertToUnsafeID);
      } else if (typeof value === 'object' && value !== null) {
        acc[key] = convertGqlIdValuesToUnsafe(value);
      } else {
        acc[key] = convertToUnsafeID(value);
      }

      return acc;
    },
    {} as T,
  );
}

/**
 * Converts all ids in a graphql data response to the safe-id pattern to ensure
 * unique ids. Mutates the object in place instead of returning a new one in
 * order to avoid performance costs of duplicating large objects in memory.
 * Requires that all nodes with ids have also requested __typename. Works
 * recursively through the data structure to convert all ids.
 *
 * @param target a node/object from a gql response
 * @throws if any ids are missing an accompanying __typename
 */
export function convertGqlNodeIdsToSafe(target: GraphQLResponseNode): void {
  if ('id' in target && target.id !== null) {
    if (!target.__typename) {
      throw new Error(`Encountered id without __typename: ${target.id}`);
    }

    // coalesce all empty objects (which should be nulls instead) into the null
    // object for that type
    target.id = convertToSafeID(
      convertToSafeTypename(target.__typename),
      target.id || GQL_GARBAGE_OBJECT_ID,
    );
  }

  Object.values(target).forEach((value) => {
    // because JS thinks typeof null === 'object'
    if (typeof value === 'object' && value !== null) {
      convertGqlNodeIdsToSafe(value);
    }
  });
}

/**
 * Normalizes polymorphic types from response Objects back to a "safe" type that
 * must be used to construct an ID for an Object type that is expected to be
 * polymorphic but has no type associated with it when obtained. An example
 * being when we use an ID parsed from a URL to query for a specific Object.
 *
 * @param typename the __typename of the received node
 * @returns safe typename
 */
export function convertToSafeTypename(typename: string): string {
  // event is the only polymorphic type we've encountered thus far, but more
  // such type normalizations could be added in subsequent if-else blocks
  if (/event/i.test(typename)) {
    return GQL_SAFE_EVENT_TYPE;
  }

  return typename;
}

/**
 * Converts ids from safe to unsafe for use in actual queries. Noops on values
 * that don't match the safe pattern or are not strings.
 *
 * @param id an id that might be in the safe pattern
 * @returns unsafe id
 */
export function convertToUnsafeID(id: string): string;
export function convertToUnsafeID<T>(id: T): T;
// eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
export function convertToUnsafeID(id: any): any {
  return GQL_SAFE_ID_REGEX.test(id) ? id.replace(GQL_SAFE_ID_REGEX, '$1') : id;
}

/**
 * Converts unsafe ids to safe by combining with typename and a custom joining
 * sequence. Should not be used directly, but instead via the entity-specific
 * helpers.
 *
 * @param typename the __typename of the entity
 * @param id the id of the entity
 * @returns safe id
 */
export function convertToSafeID(type: string, id: string): string {
  return `${type}${GQL_SAFE_ID_SEPARATOR}${id}`;
}

/**
 * Convenience method for creating safe video ids.
 *
 * @param id the id of the video
 * @returns safe video id
 */
export function convertToSafeVideoID(id: string): string {
  return convertToSafeID('Video', id);
}

/**
 * Convenience method for creating safe user ids.
 *
 * @param id the id of the user
 * @returns safe user id
 */
export function convertToSafeUserID(id: string): string {
  return convertToSafeID('User', id);
}

/**
 * Convenience method for creating safe game ids.
 *
 * @param id the id of the game
 * @returns safe game id
 */
export function convertToSafeGameID(id: string): string {
  return convertToSafeID('Game', id);
}

/**
 * Convenience method for testing if an object is a valid GQL response
 *
 * @param object  the object to test
 * @returns boolean indicating whether the object is not a GQL garbage object
 */
export function isValidObject<Obj extends GraphQLNode>(
  object: Obj | null | undefined,
): object is Obj {
  return !!object?.id && convertToUnsafeID(object.id) !== GQL_GARBAGE_OBJECT_ID;
}
