import { datatype, lorem } from 'faker';
import { randomId, randomType } from 'tachyon-test-utils';
import { garbageId, validId } from '../testHelpers';
import {
  GQL_GARBAGE_OBJECT_ID,
  GQL_SAFE_EVENT_TYPE,
  convertGqlIdValuesToUnsafe,
  convertGqlNodeIdsToSafe,
  convertGqlQueryToIncludeTypename,
  convertToSafeID,
  convertToSafeTypename,
  convertToUnsafeID,
  isValidObject,
} from '.';

describe('idConversion', () => {
  describe(convertGqlQueryToIncludeTypename, () => {
    /**
     * Generates a GQL query string for use in testing.
     */
    function testQuery(strings: TemplateStringsArray, ...keys: any[]): string {
      const body = strings
        .map((str, idx) => (idx ? `${keys[idx - 1]}${str}` : str))
        .join('');
      return `query {${body}}`;
    }

    it('annotates standalone id attribute with proper whitespace and non-safe-pattern ids are left intact', () => {
      const initialQuery = testQuery`
        video(id: "1") {
          id
        }
      `;
      const finalQuery = testQuery`
        video(id: "1") {
          id
          __typename
        }
      `;
      expect(convertGqlQueryToIncludeTypename(initialQuery)).toEqual(
        finalQuery,
      );
    });

    it('annotates id attribute in the middle of other attributes', () => {
      const initialQuery = testQuery`
        video(id: "1") {
          createdAt
          id
          deletedAt
        }
      `;
      const finalQuery = testQuery`
        video(id: "1") {
          createdAt
          id
          __typename
          deletedAt
        }
      `;
      expect(convertGqlQueryToIncludeTypename(initialQuery)).toEqual(
        finalQuery,
      );
    });

    it('annotates multiple nested id attributes', () => {
      const initialQuery = testQuery`
        video(id: "1") {
          id
          game {
            id
          }
        }
      `;
      const finalQuery = testQuery`
        video(id: "1") {
          id
          __typename
          game {
            id
            __typename
          }
        }
      `;
      expect(convertGqlQueryToIncludeTypename(initialQuery)).toEqual(
        finalQuery,
      );
    });
  });

  describe(convertGqlIdValuesToUnsafe, () => {
    it('does not convert value that does not match safe pattern', () => {
      const vars = {
        id: randomId(),
      };
      expect(convertGqlIdValuesToUnsafe(vars)).toEqual(vars);
    });

    it('converts safe-pattern values to unsafe', () => {
      const id = randomId();
      const initialVars = {
        id: convertToSafeID(randomType(), id),
      };
      const finalVars = {
        id,
      };
      expect(convertGqlIdValuesToUnsafe(initialVars)).toEqual(finalVars);
    });

    it('works correctly with multiple safe-pattern and non-id values', () => {
      const id1 = randomId();
      const id2 = randomId();
      const nonId1 = datatype.number();
      const nondId2 = lorem.word();
      const initialVars = {
        id1: convertToSafeID(randomType(), id1),
        id2: convertToSafeID(randomType(), id2),
        nonId1,
        nondId2,
      };
      const finalVars = {
        id1,
        id2,
        nonId1,
        nondId2,
      };
      expect(convertGqlIdValuesToUnsafe(initialVars)).toEqual(finalVars);
    });

    it('works recursively on nested objects', () => {
      const id = randomId();
      const initialVars = {
        nest: {
          id: convertToSafeID(randomType(), id),
        },
      };
      const finalVars = {
        nest: {
          id,
        },
      };
      expect(convertGqlIdValuesToUnsafe(initialVars)).toEqual(finalVars);
    });

    it('works on arrays', () => {
      const id1 = randomId();
      const id2 = randomId();
      const initialVars = {
        arr: [
          convertToSafeID(randomType(), id1),
          convertToSafeID(randomType(), id2),
        ],
      };
      const finalVars = {
        arr: [id1, id2],
      };
      expect(convertGqlIdValuesToUnsafe(initialVars)).toEqual(finalVars);
    });
  });

  describe(convertGqlNodeIdsToSafe, () => {
    it('does nothing when there is no id key in object', () => {
      const initialNode = {
        number: datatype.number(),
        word: lorem.word(),
      };
      const finalNode = { ...initialNode };

      convertGqlNodeIdsToSafe(initialNode as any);
      expect(initialNode).toEqual(finalNode);
    });

    it('does nothing when id is null', () => {
      const initialNode = {
        id: null,
      };
      const finalNode = { ...initialNode };

      convertGqlNodeIdsToSafe(initialNode);
      expect(initialNode).toEqual(finalNode);
    });

    it('converts to garbage object id when id is empty string', () => {
      const __typename = lorem.word();
      const initialNode = {
        __typename,
        id: '',
      };
      const finalNode = {
        ...initialNode,
        id: convertToSafeID(__typename, GQL_GARBAGE_OBJECT_ID),
      };

      convertGqlNodeIdsToSafe(initialNode);
      expect(initialNode).toEqual(finalNode);
    });

    it('throws an error if it encounters an id without an accompanying __typename', () => {
      const node = { id: lorem.word() };

      expect(() => {
        convertGqlNodeIdsToSafe(node);
      }).toThrow(/Encountered id without __typename:/);
    });

    it('converts an id with a __typename', () => {
      const __typename = lorem.word();
      const id = lorem.word();
      const initialNode = { __typename, id };
      const finalNode = {
        __typename,
        id: convertToSafeID(__typename, id),
      };

      convertGqlNodeIdsToSafe(initialNode);
      expect(initialNode).toEqual(finalNode);
    });

    it('works recursively and does not choke on nulls', () => {
      const __typename1 = lorem.word();
      const __typename2 = lorem.word();
      const id1 = lorem.word();
      const id2 = lorem.word();
      const node = {
        __typename: __typename1,
        id: id1,
        nested: {
          __typename: __typename2,
          id: id2,
          subnest: {
            random: {},
            trap2: null,
          },
        },
        trap: null,
      };
      const finalNode = {
        __typename: __typename1,
        id: convertToSafeID(__typename1, id1),
        nested: {
          __typename: __typename2,
          id: convertToSafeID(__typename2, id2),
          subnest: {
            random: {},
            trap2: null,
          },
        },
        trap: null,
      };

      convertGqlNodeIdsToSafe(node);
      expect(node).toEqual(finalNode);
    });
  });

  describe(convertToSafeTypename, () => {
    it('converts events into a single safe type', () => {
      expect(convertToSafeTypename('EventLeaf')).toEqual(GQL_SAFE_EVENT_TYPE);
      expect(convertToSafeTypename('EventCollection')).toEqual(
        GQL_SAFE_EVENT_TYPE,
      );
      expect(convertToSafeTypename('SingleEvent')).toEqual(GQL_SAFE_EVENT_TYPE);
      expect(convertToSafeTypename('PremiereEvent')).toEqual(
        GQL_SAFE_EVENT_TYPE,
      );
    });

    it('does not convert non-special-cased types', () => {
      expect(convertToSafeTypename('EvenHanded')).toEqual('EvenHanded');
    });
  });

  describe(convertToUnsafeID, () => {
    it('converts strings that match the safe pattern', () => {
      const id = randomId();
      expect(convertToUnsafeID(convertToSafeID(randomType(), id))).toEqual(id);
    });

    it('does not alter strings that do not match the safe pattern', () => {
      const id = randomId();
      expect(convertToUnsafeID(id)).toEqual(id);
    });

    it('does not error on non-string types', () => {
      const func = () => true;
      expect(convertToUnsafeID(func)).toEqual(func);

      const obj = { object: true };
      expect(convertToUnsafeID(obj)).toEqual(obj);

      const arr = [true];
      expect(convertToUnsafeID(arr)).toEqual(arr);

      const num = datatype.number();
      expect(convertToUnsafeID(num)).toEqual(num);

      expect(convertToUnsafeID(false)).toEqual(false);

      expect(convertToUnsafeID(null)).toBeNull();

      expect(convertToUnsafeID(undefined)).toBeUndefined();
    });
  });

  describe(isValidObject, () => {
    it('recognizes valid objects', () => {
      const id = validId();
      expect(isValidObject({ id })).toBeTruthy();
    });

    it('fails garbage ids', () => {
      const id = garbageId();
      expect(isValidObject({ id })).toBeFalsy();
    });

    it('fails missing objects', () => {
      expect(isValidObject(null)).toBeFalsy();
    });
  });
});
