import { GraphQLErrorType } from '../../../types';
import { logError } from '../logError';

/**
 * Tracks and logs request errors with the necessary error
 * condition.
 *
 * Use this method to generate logs that will be picked up by filters and
 * tracked as request errors with our dependency.
 *
 * @param error The Error object that was generated from fetch.
 */
export function trackAndGenerateRequestError(details: string): Error {
  logError({
    details,
    errorCondition: GraphQLErrorType.RequestError,
  });
  return new Error(GraphQLErrorType.RequestError);
}

/**
 * Tracks and logs connection errors with the necessary error
 * condition.
 *
 * Use this method to generate logs that will be picked up by filters and
 * tracked as connection errors with our dependency.
 *
 * @param error The Error object that was generated from fetch.
 */
export function trackAndGenerateConnectionError(error: unknown): Error {
  logError({
    details:
      (typeof error === 'object' && (error as Error)?.message) ||
      'Error message missing',
    errorCondition: GraphQLErrorType.ConnectionError,
  });
  return new Error(GraphQLErrorType.ConnectionError);
}

/**
 * Tracks and logs undefined response errors with the necessary error
 * condition.
 *
 * Use this method to generate logs that will be picked up by filters and
 * tracked as undefined errors with our dependency.
 *
 * @returns error type for throwing
 */
export function trackAndGenerateUndefinedResponseError(): Error {
  logError({
    details: 'Unexpected undefined response',
    errorCondition: GraphQLErrorType.UndefinedResponse,
  });
  return Error(GraphQLErrorType.UndefinedResponse);
}
