import { lorem } from 'faker';
import { GraphQLErrorType } from '../../../types';
import { logError } from '../logError';
import {
  trackAndGenerateConnectionError,
  trackAndGenerateRequestError,
  trackAndGenerateUndefinedResponseError,
} from '.';

jest.mock('../logError', () => ({
  logError: jest.fn(),
}));
const mockLogError = logError as jest.Mock;

describe('trackAndGenerateErrors', () => {
  beforeEach(() => {
    mockLogError.mockReset();
  });

  describe(trackAndGenerateRequestError, () => {
    const message = lorem.words();

    it('logs connection error and does not throw', () => {
      expect(() => trackAndGenerateRequestError(message)).not.toThrow();
      expect(mockLogError).toHaveBeenCalledWith({
        details: message,
        errorCondition: GraphQLErrorType.RequestError,
      });
    });

    it('returns an connection error', () => {
      const error = trackAndGenerateRequestError(message);
      expect(error.constructor).toBe(Error);
      expect(error.message).toEqual(GraphQLErrorType.RequestError);
    });
  });

  describe(trackAndGenerateConnectionError, () => {
    const message = lorem.words();

    it('logs connection error and does not throw', () => {
      expect(() =>
        trackAndGenerateConnectionError(new Error(message)),
      ).not.toThrow();
      expect(mockLogError).toHaveBeenCalledWith({
        details: expect.stringContaining(message),
        errorCondition: GraphQLErrorType.ConnectionError,
      });
    });

    it('returns an connection error', () => {
      const error = trackAndGenerateConnectionError(new Error(message));
      expect(error.constructor).toBe(Error);
      expect(error.message).toEqual(GraphQLErrorType.ConnectionError);
    });
  });

  describe(trackAndGenerateUndefinedResponseError, () => {
    it('logs undefined response error and does not throw', () => {
      expect(() => trackAndGenerateUndefinedResponseError()).not.toThrow();
      expect(mockLogError).toHaveBeenCalledWith({
        details: expect.stringContaining('undefined response'),
        errorCondition: GraphQLErrorType.UndefinedResponse,
      });
    });

    it('returns an undefined response error', () => {
      const error = trackAndGenerateUndefinedResponseError();
      expect(error.constructor).toBe(Error);
      expect(error.message).toEqual(GraphQLErrorType.UndefinedResponse);
    });
  });
});
