import { logger } from 'tachyon-logger';
import { HTTPStatusCode } from 'tachyon-type-library';
import { configuration } from '../../../configuration';
import type {
  TwitchGraphQLResponse,
  TwitchGraphQLResponseWithData,
  TwitchPayloadError,
} from '../../../types';
import { GraphQLErrorType } from '../../../types';
import { extractQueryName } from '../extractQueryName';
import { logError } from '../logError';

const REALLY_A_404 = 'missing userID';

interface AssertValidGraphQLResponseWithDataOpts {
  query: string;
  status: number;
  statusText: string;
}

/**
 * This method inspects a response from the GQL server and logs an appropriate
 * message if the response has an unsuccessfull status or an error in the
 * payload. It will then raise an exception in the case of an error.
 *
 * @param json The JSON in the response.
 * @param query The operation requested to be performed. We log the query for analysis.
 * @param status The status code (e.g. 200) from the response.
 * @param statusText The textual representation of the status code.
 */
export function assertValidGraphQLResponseWithData(
  json: TwitchGraphQLResponse,
  { query, status, statusText }: AssertValidGraphQLResponseWithDataOpts,
): asserts json is TwitchGraphQLResponseWithData {
  if (status < 200 || status > 299) {
    const errorCondition =
      status === HTTPStatusCode.Unauthorized
        ? GraphQLErrorType.HTTPStatusUnauthorized
        : GraphQLErrorType.HTTPStatusError;

    logError({
      details: statusText,
      errorCondition,
      status,
    });
    throw new Error(errorCondition);
  }

  if (!json || typeof json !== 'object') {
    logError({
      details: json ? JSON.stringify(json) : 'missing json',
      errorCondition: GraphQLErrorType.InvalidResponseBody,
    });
    throw new Error(GraphQLErrorType.InvalidResponseBody);
  }

  if (!('data' in json) || !json.data || typeof json.data !== 'object') {
    // if not data due to errors, give the more detailed error message
    if (json.errors) {
      logError({
        details: JSON.stringify(json.errors),
        errorCondition: GraphQLErrorType.ErrorResponseInBody,
      });
      throw new Error(GraphQLErrorType.ErrorResponseInBody);
    }

    logError({
      details: JSON.stringify(json),
      errorCondition: GraphQLErrorType.InvalidResponseBody,
    });
    throw new Error(GraphQLErrorType.InvalidResponseBody);
  }

  if (json.errors && !reallyA404(json.errors)) {
    if (configuration.debug) {
      logger.debug({
        category: 'assertValidGraphQLResponseWithData',
        context: {
          errors: JSON.stringify(json.errors),
        },
        message: 'Received GQL errors array',
        package: 'tachyon-relay',
      });
    }

    if (
      configuration.errorsArrayIsFatal(json.errors, extractQueryName(query))
    ) {
      logError({
        details: JSON.stringify(json.errors),
        errorCondition: GraphQLErrorType.ErrorResponseInBody,
      });
      throw new Error(GraphQLErrorType.ErrorResponseInBody);
    }
  }
}

/**
 * This method checks whether the errors GQL responded with should be treated
 * as a 404. This is necessary because many VOD-related parts of the schema
 * when accessed from a nonexistent user generate erroneous errors.
 *
 * @param errors The errors from a GQL response.
 * @returns true if the errors can be classified as a 404, false otherwise.
 */
function reallyA404(errors: TwitchPayloadError[]): boolean {
  return !!errors.every((error) => error.message === REALLY_A_404);
}
