import { configuration, resetConfiguration } from '../../../configuration';
import type { TwitchGraphQLResponse } from '../../../types';
import { GraphQLErrorType } from '../../../types';
import { logError } from '../logError';
import { assertValidGraphQLResponseWithData } from '.';

jest.mock('../logError', () => ({
  logError: jest.fn(),
}));
const mockLogError = logError as jest.Mock;

describe(assertValidGraphQLResponseWithData, () => {
  const query = 'I AM A GQL QUERY!';

  beforeEach(() => {
    mockLogError.mockReset();
  });

  afterEach(() => {
    resetConfiguration();
  });

  describe('when a status code is non-200', () => {
    it('logs the right error', () => {
      const response500 = {
        status: 500,
        statusText: 'oh noes!',
      };

      expect(() => {
        assertValidGraphQLResponseWithData({} as TwitchGraphQLResponse, {
          query,
          status: response500.status,
          statusText: response500.statusText,
        });
      }).toThrow(GraphQLErrorType.HTTPStatusError);

      expect(mockLogError).toHaveBeenCalledWith({
        details: response500.statusText,
        errorCondition: GraphQLErrorType.HTTPStatusError,
        status: response500.status,
      });
    });

    it('special cases 401 Unauthorized', () => {
      const response401 = {
        status: 401,
        statusText: 'oh noes!',
      };

      expect(() => {
        assertValidGraphQLResponseWithData({} as TwitchGraphQLResponse, {
          query,
          status: response401.status,
          statusText: response401.statusText,
        });
      }).toThrow(GraphQLErrorType.HTTPStatusUnauthorized);

      expect(mockLogError).toHaveBeenCalledWith({
        details: response401.statusText,
        errorCondition: GraphQLErrorType.HTTPStatusUnauthorized,
        status: response401.status,
      });
    });
  });

  describe('when the response payload is invalid', () => {
    it('handles missing json', () => {
      const responseWithoutJson = {
        json: undefined,
        status: 200,
        statusText: 'OK',
      };

      expect(() => {
        assertValidGraphQLResponseWithData(responseWithoutJson.json as any, {
          query,
          status: responseWithoutJson.status,
          statusText: responseWithoutJson.statusText,
        });
      }).toThrow(GraphQLErrorType.InvalidResponseBody);

      expect(mockLogError).toHaveBeenCalledWith({
        details: 'missing json',
        errorCondition: GraphQLErrorType.InvalidResponseBody,
      });
    });

    it('handles non-object json value', () => {
      const responseWithNonObjectJson = {
        json: 99,
        status: 200,
        statusText: 'OK',
      };

      expect(() => {
        assertValidGraphQLResponseWithData(
          responseWithNonObjectJson.json as any,
          {
            query,
            status: responseWithNonObjectJson.status,
            statusText: responseWithNonObjectJson.statusText,
          },
        );
      }).toThrow(GraphQLErrorType.InvalidResponseBody);

      expect(mockLogError).toHaveBeenCalledWith({
        details: '99',
        errorCondition: GraphQLErrorType.InvalidResponseBody,
      });
    });

    it('handles missing json.data', () => {
      const responseWithoutJsonData = {
        json: {},
        status: 200,
        statusText: 'OK',
      };

      expect(() => {
        assertValidGraphQLResponseWithData(
          responseWithoutJsonData.json as TwitchGraphQLResponse,
          {
            query,
            status: responseWithoutJsonData.status,
            statusText: responseWithoutJsonData.statusText,
          },
        );
      }).toThrow(GraphQLErrorType.InvalidResponseBody);

      expect(mockLogError).toHaveBeenCalledWith({
        details: '{}',
        errorCondition: GraphQLErrorType.InvalidResponseBody,
      });
    });

    it('handles non-object json.data value', () => {
      const responseWithNonObjectJsonData = {
        json: { data: 'data' },
        status: 200,
        statusText: 'OK',
      };

      expect(() => {
        assertValidGraphQLResponseWithData(
          responseWithNonObjectJsonData.json as any,
          {
            query,
            status: responseWithNonObjectJsonData.status,
            statusText: responseWithNonObjectJsonData.statusText,
          },
        );
      }).toThrow(GraphQLErrorType.InvalidResponseBody);

      expect(mockLogError).toHaveBeenCalledWith({
        details: '{"data":"data"}',
        errorCondition: GraphQLErrorType.InvalidResponseBody,
      });
    });

    it('handles missing json.data with accompanying errors', () => {
      const responseWithoutJsonDataWithErrors = {
        json: {
          errors: [{ message: 'i am an error', path: ['here'] }],
        },
        status: 200,
        statusText: 'OK',
      };

      expect(() => {
        assertValidGraphQLResponseWithData(
          responseWithoutJsonDataWithErrors.json as TwitchGraphQLResponse,
          {
            query,
            status: responseWithoutJsonDataWithErrors.status,
            statusText: responseWithoutJsonDataWithErrors.statusText,
          },
        );
      }).toThrow(GraphQLErrorType.ErrorResponseInBody);

      expect(mockLogError).toHaveBeenCalledWith({
        details: JSON.stringify(responseWithoutJsonDataWithErrors.json.errors),
        errorCondition: GraphQLErrorType.ErrorResponseInBody,
      });
    });
  });

  describe('when the JSON has an `errors` section', () => {
    describe('for non-special case errors', () => {
      const responseWithErrors = {
        json: {
          data: {},
          errors: [{ message: 'i am an error', path: ['here'] }],
        },
        status: 200,
        statusText: 'OK',
      };

      it('logs the right error', () => {
        expect(() => {
          assertValidGraphQLResponseWithData(responseWithErrors.json, {
            query,
            status: responseWithErrors.status,
            statusText: responseWithErrors.statusText,
          });
        }).toThrow(GraphQLErrorType.ErrorResponseInBody);

        expect(mockLogError).toHaveBeenCalledWith({
          details: JSON.stringify(responseWithErrors.json.errors),
          errorCondition: GraphQLErrorType.ErrorResponseInBody,
        });
      });

      it('does not log error if config override returns false', () => {
        configuration.errorsArrayIsFatal = () => false;
        expect(() => {
          assertValidGraphQLResponseWithData(responseWithErrors.json, {
            query,
            status: responseWithErrors.status,
            statusText: responseWithErrors.statusText,
          });
        }).not.toThrow();

        expect(mockLogError).not.toHaveBeenCalled();
      });
    });

    describe('for special case "missing userID" errors', () => {
      it('logs the right error when only some errors are 404s', () => {
        const responseMixedErrors = {
          json: {
            data: {},
            errors: [
              { message: 'missing userID', path: ['here'] },
              { message: 'missing userIBrie', path: ['digital cheese'] },
            ],
          },
          status: 200,
          statusText: 'OK',
        };

        expect(() => {
          assertValidGraphQLResponseWithData(responseMixedErrors.json, {
            query,
            status: responseMixedErrors.status,
            statusText: responseMixedErrors.statusText,
          });
        }).toThrow(GraphQLErrorType.ErrorResponseInBody);

        expect(mockLogError).toHaveBeenCalledWith({
          details: JSON.stringify(responseMixedErrors.json.errors),
          errorCondition: GraphQLErrorType.ErrorResponseInBody,
        });
      });

      it('ignores the case that all the errors are bad 404s', () => {
        const responseAllErrorsAre404s = {
          json: {
            data: {},
            errors: [
              { message: 'missing userID', path: ['here'] },
              { message: 'missing userID', path: ['over here'] },
            ],
          },
          status: 200,
          statusText: 'OK',
        };

        expect(() => {
          assertValidGraphQLResponseWithData(responseAllErrorsAre404s.json, {
            query,
            status: responseAllErrorsAre404s.status,
            statusText: responseAllErrorsAre404s.statusText,
          });
        }).not.toThrow();

        expect(mockLogError).not.toHaveBeenCalled();
      });
    });
  });
});
