import type { RequestParameters, Variables } from 'relay-runtime';
import { logger } from 'tachyon-logger';
import { getAuthorizationHeader } from '../../authorizationHeader';
import { configuration } from '../../configuration';
import {
  convertGqlIdValuesToUnsafe,
  convertGqlNodeIdsToSafe,
  convertGqlQueryToIncludeTypename,
} from '../../idConversion';
import type {
  RelayEvent,
  TwitchGraphQLResponse,
  TwitchGraphQLResponseWithData,
  TwitchPayloadError,
} from '../../types';
import {
  assertValidGraphQLResponseWithData,
  trackAndGenerateConnectionError,
  trackAndGenerateUndefinedResponseError,
} from '../errorTrackers';
import { logServicesFromResponse } from '../logServicesFromResponse';

export type FetchAndValidateOpts = {
  authorization?:
    | {
        token: Parameters<typeof getAuthorizationHeader>[0];
        unauthorizedHandler: () => void;
      }
    | undefined;
  // Device ID is not passed on the server for pages that are noncached, and is
  // not passed from the RequestInfo process
  deviceId?: string | undefined;
  /**
   * Extra headers added to the GQL fetch request (will not override any headers
   * normally set by the library)
   */
  gqlHeaders?: Record<string, string> | undefined;
  // Language is not passed from the RequestInfo process
  language?: string | undefined;
  /**
   * Handler invoked when the `errors` array has entries but the GQL request did not throw
   */
  nonFatalErrorsHandler?: ((errors: TwitchPayloadError[]) => void) | undefined;
  /**
   * For reporting events related to Relay fetches
   */
  onEvent?: ((event: RelayEvent) => void) | undefined;
  query: NonNullable<RequestParameters['text']>;
  variables: Variables;
};

// this can probably be split into 2 smaller pieces that are easier to test
export async function fetchAndValidate({
  authorization,
  deviceId,
  gqlHeaders,
  language,
  nonFatalErrorsHandler,
  query,
  variables,
}: FetchAndValidateOpts): Promise<TwitchGraphQLResponseWithData> {
  if (!configuration.clientId) {
    logger.error({
      category: 'fetchAndValidate',
      message:
        'Relay fetch attempted without clientId. Use configureTachyonRelay.',
      package: 'tachyon-relay',
    });
  }

  let response: Response | undefined;
  try {
    const headers: Record<string, string> = {
      ...gqlHeaders,
      /* The Twitch GQL Server assumes a value of application/json, regardless of the header value */
      Accept: 'application/json',
      'Accept-Language': language ?? 'en-US',
      'Client-Id': configuration.clientId,
      /* The Twitch GQL Server assumes this value to always be application/json */
      'Content-Type': 'application/json',
    };

    const authorizationHeader = getAuthorizationHeader(authorization?.token);
    if (authorizationHeader) {
      headers['Authorization'] = authorizationHeader;
    }

    if (deviceId) {
      headers['Device-Id'] = deviceId;
    }

    if (configuration.debug) {
      headers['Twitch-Trace'] = 'extensions';

      if (configuration.failServices?.length) {
        headers['Fail'] = configuration.failServices.join(',');
      }
    }

    response = await fetch(configuration.gqlEndpoint, {
      agent: configuration.serverHttpsAgent,
      body: JSON.stringify({
        query: convertGqlQueryToIncludeTypename(query),
        variables: convertGqlIdValuesToUnsafe(variables),
      }),
      headers,
      method: 'POST',
      timeout: configuration.serverFetchTimeout,
    } as RequestInit);
  } catch (error) {
    throw trackAndGenerateConnectionError(error);
  }

  if (response !== undefined) {
    const graphQLResponse: TwitchGraphQLResponse = await response.json();
    assertValidGraphQLResponseWithData(graphQLResponse, {
      query,
      status: response.status,
      statusText: response.statusText,
    });
    convertGqlNodeIdsToSafe(graphQLResponse.data);

    if (configuration.debug) {
      logServicesFromResponse(graphQLResponse);
    }

    if (graphQLResponse.errors) {
      nonFatalErrorsHandler?.(graphQLResponse.errors);
    }

    // functionality based on gql response extensions
    // consider exposing this whole thing as a onExtension-type callback instead
    // of having secret side-effects in the package itself
    if (graphQLResponse.extensions) {
      const { durationMilliseconds } = graphQLResponse.extensions;

      // log query times on server for monitoring in cloudwatch
      if (durationMilliseconds && typeof window === 'undefined') {
        const queryMatch = query.match(/query ([\w_]+)\(/);

        logger.info({
          category: 'gqlQueryMeta',
          context: {
            duration: graphQLResponse.extensions.durationMilliseconds,
            query: queryMatch?.[1] ?? 'Unknown',
          },
          // exact string tied to cloudwatch metric filters
          message: 'gql query timing',
          package: 'tachyon-relay',
        });
      }
    }

    return graphQLResponse;
  }
  throw trackAndGenerateUndefinedResponseError();
}
