import type {
  FetchFunction,
  GraphQLResponse,
  RequestParameters,
  Variables,
} from 'relay-runtime';
import { isBrowser } from 'tachyon-utils-stdlib';
import { configuration } from '../../configuration';
import { GraphQLErrorType } from '../../types';
import { trackAndGenerateRequestError } from '../errorTrackers';
import type { FetchAndValidateOpts } from '../fetchAndValidate';
import { fetchAndValidate } from '../fetchAndValidate';

export type FetchQueryOpts = Pick<
  FetchAndValidateOpts,
  | 'authorization'
  | 'deviceId'
  | 'gqlHeaders'
  | 'language'
  | 'nonFatalErrorsHandler'
  | 'onEvent'
>;

export function getFetchQuery({
  authorization,
  deviceId,
  gqlHeaders,
  language,
  nonFatalErrorsHandler,
  onEvent,
}: FetchQueryOpts = {}): FetchFunction {
  // Define a function that fetches the results of an operation (query/mutation/etc)
  // and returns its results as a Promise:
  return async function fetchQuery(
    { text: query }: RequestParameters,
    variables: Variables = {},
    _cacheConfig: unknown,
    _uploadables?: unknown,
  ): Promise<GraphQLResponse> {
    if (!query) {
      throw trackAndGenerateRequestError(
        'Missing field "text" in "fetchQuery" request parameters',
      );
    }

    const opts: FetchAndValidateOpts = {
      authorization,
      deviceId,
      gqlHeaders,
      language,
      nonFatalErrorsHandler,
      onEvent,
      query,
      variables,
    };

    try {
      return await fetchAndValidate(opts);
    } catch (error) {
      if (
        error instanceof Error &&
        error.message === GraphQLErrorType.HTTPStatusUnauthorized &&
        authorization
      ) {
        authorization.unauthorizedHandler();
        // Retry the request without the authorization token after encourntering an error from an invalid token.
        // This improves the DX for consuming clients so they don't need to mutate the authorization options passed
        // into this package after encourntering an unauthorized error.
        const { authorization: _removed, ...optionsWithoutAuth } = opts;
        return await fetchAndValidate(optionsWithoutAuth);
      }

      if (isBrowser() && configuration.browserRetry) {
        return await fetchAndValidate(opts);
      }

      throw error;
    }
  };
}
