import { datatype, random } from 'faker';
import type { RequestParameters, Variables } from 'relay-runtime';
import { configuration, resetConfiguration } from '../../configuration';
import { GraphQLErrorType } from '../../types';
import { fetchAndValidate } from '../fetchAndValidate';
import type { FetchQueryOpts } from '.';
import { getFetchQuery } from '.';

jest.mock('tachyon-logger', () => ({ logger: { error: jest.fn() } }));
let mockIsBrowser = false;
jest.mock('tachyon-utils-stdlib', () => ({
  isBrowser: () => mockIsBrowser,
}));
jest.mock('../fetchAndValidate', () => ({
  fetchAndValidate: jest.fn(),
}));
const mockFetchAndValidate = fetchAndValidate as jest.Mock;

describe(getFetchQuery, () => {
  beforeEach(() => {
    mockIsBrowser = false;
    mockFetchAndValidate.mockReset();
  });

  afterEach(() => {
    resetConfiguration();
  });

  const fetchQueryOpts: FetchQueryOpts = {
    deviceId: datatype.uuid(),
    gqlHeaders: { [random.word()]: random.words() },
    language: random.locale(),
    onEvent: () => undefined,
  };
  const requestParams = { text: 'text' } as RequestParameters;
  const requestVars = { variable: true } as Variables;
  const jsonResponse = { response: true };

  const fetchQuery = getFetchQuery(fetchQueryOpts);

  it('throws error when called with request parameters missing text field', async () => {
    await expect(fetchQuery({} as RequestParameters, {}, {})).rejects.toThrow(
      GraphQLErrorType.RequestError,
    );
  });

  it('returns successful fetch on the server and passes correct config', async () => {
    mockIsBrowser = false;
    mockFetchAndValidate.mockReturnValueOnce(jsonResponse);

    await expect(fetchQuery(requestParams, requestVars, {})).resolves.toEqual(
      jsonResponse,
    );
    expect(mockFetchAndValidate).toHaveBeenCalledWith({
      ...fetchQueryOpts,
      query: requestParams.text,
      variables: requestVars,
    });
  });

  it('errors and does not retry failed fetch on the server', async () => {
    mockIsBrowser = false;
    mockFetchAndValidate.mockImplementationOnce(() => {
      throw new Error(GraphQLErrorType.ConnectionError);
    });

    await expect(fetchQuery(requestParams, {}, {})).rejects.toThrow(
      GraphQLErrorType.ConnectionError,
    );
  });

  it('errors and does not retry failed fetch in browser with config override', async () => {
    mockIsBrowser = true;
    configuration.browserRetry = false;
    mockFetchAndValidate.mockImplementationOnce(() => {
      throw new Error(GraphQLErrorType.ConnectionError);
    });

    await expect(fetchQuery(requestParams, {}, {})).rejects.toThrow(
      GraphQLErrorType.ConnectionError,
    );
  });

  it('returns successful fetch in browser and passes correct config', async () => {
    mockIsBrowser = true;
    mockFetchAndValidate.mockReturnValueOnce(jsonResponse);

    await expect(fetchQuery(requestParams, requestVars, {})).resolves.toEqual(
      jsonResponse,
    );
    expect(mockFetchAndValidate).toHaveBeenCalledWith({
      ...fetchQueryOpts,
      query: requestParams.text,
      variables: requestVars,
    });
  });

  it('retries and returns successful 2nd fetch in browser after initial failure', async () => {
    mockIsBrowser = true;
    mockFetchAndValidate
      .mockImplementationOnce(() => {
        throw new Error(GraphQLErrorType.ConnectionError);
      })
      .mockImplementationOnce(() => jsonResponse);

    await expect(fetchQuery(requestParams, requestVars, {})).resolves.toEqual(
      jsonResponse,
    );
  });

  it('errors and does not rety 2 failed fetches in browser', async () => {
    mockIsBrowser = true;
    mockFetchAndValidate.mockImplementation(() => {
      throw new Error(GraphQLErrorType.ConnectionError);
    });

    await expect(fetchQuery(requestParams, {}, {})).rejects.toThrow(
      GraphQLErrorType.ConnectionError,
    );
  });

  it('invokes the "unauthorizedHandler" and retries the request without authorization when 401 failure in browser', async () => {
    const unauthorizedHandler = jest.fn();
    const authorization = {
      token: 'foo',
      unauthorizedHandler,
    };
    const query = getFetchQuery({
      authorization,
      ...fetchQueryOpts,
    });
    mockIsBrowser = true;
    mockFetchAndValidate
      .mockImplementationOnce(() => {
        throw new Error(GraphQLErrorType.HTTPStatusUnauthorized);
      })
      .mockImplementationOnce(() => jsonResponse);

    await expect(query(requestParams, requestVars, {})).resolves.toEqual(
      jsonResponse,
    );

    expect(unauthorizedHandler).toHaveBeenCalledTimes(1);
    expect(mockFetchAndValidate).toHaveBeenCalledTimes(2);
    expect(mockFetchAndValidate.mock.calls[0][0].authorization).toEqual(
      authorization,
    );
    expect(mockFetchAndValidate.mock.calls[1][0].authorization).toBeUndefined();
  });

  it('invokes the "unauthorizedHandler" and retries the request without authorization when 401 failure on server', async () => {
    const unauthorizedHandler = jest.fn();
    const authorization = {
      token: 'foo',
      unauthorizedHandler,
    };
    const query = getFetchQuery({
      authorization,
      ...fetchQueryOpts,
    });
    mockIsBrowser = false;
    mockFetchAndValidate
      .mockImplementationOnce(() => {
        throw new Error(GraphQLErrorType.HTTPStatusUnauthorized);
      })
      .mockImplementationOnce(() => jsonResponse);

    await expect(query(requestParams, requestVars, {})).resolves.toEqual(
      jsonResponse,
    );

    expect(unauthorizedHandler).toHaveBeenCalledTimes(1);
    expect(mockFetchAndValidate).toHaveBeenCalledTimes(2);
    expect(mockFetchAndValidate.mock.calls[0][0].authorization).toEqual(
      authorization,
    );
    expect(mockFetchAndValidate.mock.calls[1][0].authorization).toBeUndefined();
  });
});
