import { Environment, Network, RecordSource, Store } from 'relay-runtime';
import { isBrowser } from 'tachyon-utils-stdlib';
import { configuration } from '../../configuration';
import type { RecordMap } from '../../types';
import type { FetchQueryOpts } from '../getFetchQuery';
import { getFetchQuery } from '../getFetchQuery';

export type InitEnvironmentOpts = {
  fetchQueryOpts?: FetchQueryOpts | undefined;
  records?: RecordMap | undefined;
};

let relayEnvironment: Environment | undefined;

// istanbul ignore next: high cost, low value
export function initEnvironment({
  fetchQueryOpts,
  records = {},
}: InitEnvironmentOpts = {}): Environment {
  if (isBrowser() && relayEnvironment) {
    return relayEnvironment;
  }

  // Create a network layer from the fetch function
  const network = Network.create(getFetchQuery(fetchQueryOpts));
  const store = new Store(new RecordSource(records));

  // Make sure to create a new Relay environment for every server-side request
  // so that data isn't shared between independent requests
  if (typeof window === 'undefined') {
    return new Environment({
      network,
      store,
    });
  }

  relayEnvironment = new Environment({
    network,
    store,
  });

  if (isBrowser() && configuration.debug) {
    (window as any).getTachyonRelayStore = () =>
      relayEnvironment?.getStore().getSource().toJSON() || {};
  }
  return relayEnvironment;
}
