import type {
  GraphQLResponse,
  PayloadError,
  RecordSource,
  MutationConfig as RelayMutationConfig,
  MutationParameters as RelayMutationParameters,
} from 'relay-runtime';
import type { SetRequired } from 'type-fest';

// the represents the inner structure of Relay's RecordSource, which we need
// for our 404 tracking. This is brittle and may break version-to-version.
export type RecordMap = NonNullable<
  ConstructorParameters<typeof RecordSource>[0]
>;

export type GraphQLResponseNode = {
  __typename?: string;
  id?: string | null;
};

export type GraphQLNode = {
  id: string;
};

/**
 * relay-runtime's PayloadData updated for Twitch's GraphQL API
 */
type TwitchPayloadData = GraphQLResponseNode;

/**
 * relay-runtime's PayloadError updated for Twitch's GraphQL API
 */
export type TwitchPayloadError = Omit<PayloadError, 'locations'> & {
  path?: string[];
};

type ServiceQuery = {
  /**
   * End time of the backing API call, in an ISO timestamp.
   */
  end: string;
  /**
   * Details about the data resolution used for a field in the query. Comes in
   * the format `service.${service_name}.${service_method}`.
   */
  method: string;
  /**
   * Start time of the backing API call, in an ISO timestamp.
   */
  start: string;
};

/**
 * relay-runtime's PayloadExtensions updated for Twitch's GraphQL API
 */
export type TwitchPayloadExtensions = {
  durationMilliseconds?: number;
  queries?: ServiceQuery[];
  requestID?: string;
};

// extract inner types not directly exported from relay so we can alter them
type ExtractGraphQLResponseWithData<GraphQLResponseUnion> =
  GraphQLResponseUnion extends {
    errors: PayloadError[];
  }
    ? never
    : GraphQLResponseUnion;

type ExtractGraphQLResponseWithoutData<GraphQLResponseUnion> =
  GraphQLResponseUnion extends {
    errors: PayloadError[];
  }
    ? GraphQLResponseUnion
    : never;

type TwitchOverridenProperties = 'data' | 'errors' | 'extensions';

/**
 * relay-runtime's GraphQLResponseWithData updated for Twitch's GraphQL API
 */
export type TwitchGraphQLResponseWithData = Omit<
  ExtractGraphQLResponseWithData<GraphQLResponse>,
  TwitchOverridenProperties
> & {
  data: TwitchPayloadData;
  errors?: TwitchPayloadError[];
  extensions?: TwitchPayloadExtensions;
};

/**
 * relay-runtime's GraphQLResponseWithoutData updated for Twitch's GraphQL API
 */
export type TwitchGraphQLResponseWithoutData = Omit<
  ExtractGraphQLResponseWithoutData<GraphQLResponse>,
  TwitchOverridenProperties
> & {
  errors: TwitchPayloadError[];
  extensions?: TwitchPayloadExtensions;
};

/**
 * relay-runtime's GraphQLResponse updated for Twitch's GraphQL API
 */
export type TwitchGraphQLResponse =
  | TwitchGraphQLResponseWithData
  | TwitchGraphQLResponseWithoutData;

export enum GraphQLErrorType {
  ConnectionError = 'connection-error',
  ErrorResponseInBody = 'error-response-in-body',
  HTTPStatusError = 'http-status-error',
  HTTPStatusUnauthorized = 'http-status-unauthorized',
  InvalidResponseBody = 'invalid-response-body',
  RequestError = 'request-error',
  UndefinedResponse = 'undefined-response',
}

/**
 * Tracking events emitted during GraphQL fetches
 */
export type RelayEvent = { event: string };

/**
 * Relay MutationParameters that prevent variables from evaluating to {} (which
 * removes all type constraints).
 */
export type MutationParameters<MutationVariables extends {}> = Omit<
  RelayMutationParameters,
  'variables'
> & {
  readonly variables: MutationVariables;
};

/**
 * Relay MutationConfig that prevent variables from evaluating to {} (which
 * removes all type constraints) and makes the `onError` prop mandatory.
 */
export type MutationConfig<
  MutationVariables extends {},
  MP extends MutationParameters<MutationVariables> = MutationParameters<MutationVariables>,
> = SetRequired<RelayMutationConfig<MP>, 'onError'>;
