import { renderHook } from '@testing-library/react-hooks';
import { useRefetchList } from '.';

describe(useRefetchList, () => {
  it('extracts nodes and endCursor', () => {
    const query = {
      streams: {
        edges: [
          {
            cursor: '1321',
            node: {
              ' $fragmentRefs': { StreamCard_stream: true },
              id: '123',
            },
          },
        ],
      },
    };

    const { result } = renderHook(useRefetchList, {
      initialProps: query.streams.edges,
    });
    expect(result.current.endCursor).toEqual('1321');
    expect(result.current.nodes).toEqual([
      {
        ' $fragmentRefs': { StreamCard_stream: true },
        id: '123',
      },
    ]);
  });

  it('noop when called with same endCursor', () => {
    const query = {
      streams: {
        edges: [
          {
            cursor: '1321',
            node: {
              ' $fragmentRefs': { StreamCard_stream: true },
              id: '123',
            },
          },
        ],
      },
    };

    const { rerender, result } = renderHook(useRefetchList, {
      initialProps: query.streams.edges,
    });
    rerender(query.streams.edges);
    expect(result.current.endCursor).toEqual('1321');
    expect(result.current.nodes).toEqual([
      {
        ' $fragmentRefs': { StreamCard_stream: true },
        id: '123',
      },
    ]);
    expect(result.current.noMore).toEqual(false);
  });

  it('returns noMore flag when no nodes are returned', () => {
    const query = {
      streams: {
        edges: [
          {
            cursor: '1321',
            node: {
              ' $fragmentRefs': { StreamCard_stream: true },
              id: '123',
            },
          },
        ],
      },
    };

    const { rerender, result } = renderHook(useRefetchList, {
      initialProps: query.streams.edges,
    });
    rerender([
      {
        cursor: '1322',
        // @ts-expect-error: tests
        node: null,
      },
    ]);
    expect(result.current.endCursor).toEqual('1322');
    expect(result.current.nodes).toEqual([
      {
        ' $fragmentRefs': { StreamCard_stream: true },
        id: '123',
      },
    ]);
    expect(result.current.noMore).toEqual(true);
  });

  it('filters duplicates', () => {
    const query = {
      streams: {
        edges: [
          {
            cursor: '1321',
            node: {
              ' $fragmentRefs': { StreamCard_stream: true },
              id: '123',
            },
          },
          {
            cursor: '1321',
            node: {
              ' $fragmentRefs': { StreamCard_stream: true },
              id: '123',
            },
          },
        ],
      },
    };

    const { result } = renderHook(useRefetchList, {
      initialProps: query.streams.edges,
    });

    expect(result.current.endCursor).toEqual('1321');
    expect(result.current.nodes).toEqual([
      {
        ' $fragmentRefs': { StreamCard_stream: true },
        id: '123',
      },
    ]);
    expect(result.current.noMore).toEqual(false);
  });

  it('preserves previous nodes as more are fetched', () => {
    const query = {
      streams: {
        edges: [
          {
            cursor: '1321',
            node: {
              ' $fragmentRefs': { StreamCard_stream: true },
              id: '123',
            },
          },
        ],
      },
    };

    const { rerender, result } = renderHook(useRefetchList, {
      initialProps: query.streams.edges,
    });
    rerender([
      {
        cursor: '1322',
        node: {
          ' $fragmentRefs': { StreamCard_stream: true },
          id: '124',
        },
      },
    ]);
    expect(result.current.endCursor).toEqual('1322');
    expect(result.current.nodes).toEqual([
      {
        ' $fragmentRefs': { StreamCard_stream: true },
        id: '123',
      },
      {
        ' $fragmentRefs': { StreamCard_stream: true },
        id: '124',
      },
    ]);
    expect(result.current.noMore).toEqual(false);
  });
});
