import { useCallback, useRef } from 'react';
import type { RefetchFnDynamic } from 'react-relay/hooks';
import {
  fetchQuery,
  useRefetchableFragment,
  useRelayEnvironment,
} from 'react-relay/hooks';
import type { KeyType, KeyTypeData } from 'react-relay/relay-hooks/helpers';
import type { useRefetchableFragmentHookType } from 'react-relay/relay-hooks/useRefetchableFragment';
import type { Options } from 'react-relay/relay-hooks/useRefetchableFragmentNode';
import type {
  GraphQLTaggedNode,
  OperationType,
  Subscription,
  Variables,
} from 'relay-runtime';

/**
 * @experimental
 * This hook is the same as the standard useRefetchableFragment except it allows
 * consumers to avoid falling back to Suspense. This emulates future React (with
 * Concurrent mode) where the Suspense fallback can be the stale tree while
 * awaiting data. To enable this, it requires an additional parameter that is a
 * standalone query that wraps your fragment, but that is the only change to the
 * API (enabling an easy transition to proper Concurrent mode in the future).
 *
 * Inspired by: https://relay.dev/docs/guided-tour/refetching/refetching-fragments-with-different-data/#if-you-need-to-avoid-suspense
 */
// istanbul ignore next: high effort/mocking for low payoff
export function useRefetchableFragmentWithoutSuspense<
  TQuery extends OperationType,
  TKey extends KeyType,
>(
  fragmentInput: GraphQLTaggedNode,
  fragmentRef: TKey,
  queryForFragment: GraphQLTaggedNode,
): useRefetchableFragmentHookType<TQuery, TKey, KeyTypeData<TKey>> {
  const [data, _refetch] = useRefetchableFragment(fragmentInput, fragmentRef);
  const environment = useRelayEnvironment();
  const subscription = useRef<Subscription | null>(null);

  const refetch: RefetchFnDynamic<TQuery, TKey> = useCallback(
    (variables: Variables, options?: Options) => {
      if (!subscription.current) {
        subscription.current = fetchQuery(
          environment,
          queryForFragment,
          variables,
        ).subscribe({
          complete: () => {
            _refetch(variables, {
              fetchPolicy: 'store-only',
              onComplete: options?.onComplete,
            });
            subscription.current = null;
          },
          error: (e: Error) => {
            options?.onComplete?.(e);
            subscription.current = null;
          },
        });
      }

      return {
        dispose: () => {
          subscription.current?.unsubscribe();
          subscription.current = null;
        },
      };
    },
    [_refetch, environment, queryForFragment],
  );

  return [data, refetch];
}
