import type { Request, Response } from 'express';
import {
  DEFAULT_COOKIE_DOMAIN,
  MAX_COOKIE_AGE,
  SAMESITE_COMPAT,
  generateCookieOpts,
  getComplexValue,
  setComplexValue,
} from 'tachyon-utils-twitch';
import type {
  ClearCookieValueOpts,
  GetAndExtendCookieValueOpts,
  GetCookieValueOpts,
  SetCookieValueComplexOpts,
  SetCookieValueOpts,
  VersionedPayload,
} from 'tachyon-utils-twitch';

export { GLOBAL_COOKIE_NAMES } from 'tachyon-utils-twitch';

let cookieDomain = DEFAULT_COOKIE_DOMAIN;

/**
 * Sets the default domain used when setting cookies on the server (individual
 * uses of setCookieValue can still override this default setting). Original
 * value is `twitch.tv`.
 *
 * @param newDomain The domain that cookies will be set to.
 */
export function setCookieDomainOnServer(newDomain: string): void {
  cookieDomain = newDomain;
}

/**
 * Resets the default domain used when setting cookies on the server to
 * `twitch.tv` (individual of setCookieValue can still override this default
 * setting).
 */
export function resetCookieDomainOnServer(): void {
  cookieDomain = DEFAULT_COOKIE_DOMAIN;
}

export interface GetCookieOnServerOpts extends GetCookieValueOpts {
  req: Request;
}

/**
 * Gets a cookie value on a Request object serverside.
 * When retrieving a value other than a string (such as JSON / Objects) use `getCookieComplexValueOnServer`.
 *
 * NOTE: `getAndExtendCookieValueOnServer` should generally be preferred over using this utility directly.
 */
export function getCookieValueOnServer({
  name,
  req,
}: GetCookieOnServerOpts): string | undefined {
  return req.cookies[name] ?? req.cookies[name + SAMESITE_COMPAT];
}

/**
 * Identical to `getCookieValueOnServer` but is used when an object was stored instead of a string. The object is deserialized.
 *
 * NOTE: `getAndExtendCookieComplexValueOnServer` should generally be preferred over using this utility directly.
 */
export function getCookieComplexValueOnServer<Payload extends VersionedPayload>(
  opts: GetCookieOnServerOpts,
): Payload | undefined {
  return getComplexValue(getCookieValueOnServer(opts));
}

type GetAndExtendCookieOnServerOpts = GetAndExtendCookieValueOpts & {
  req: Request;
  res: Response;
};

/**
 * Gets a cookie value on a Request object serverside and extends the cookie expiration.
 *
 * When retrieving a value other than a string (such as JSON / Objects) use `getAndExtendCookieComplexValueOnServer`.
 */
export function getAndExtendCookieValueOnServer({
  extension = MAX_COOKIE_AGE,
  migrationNames = [],
  name,
  req,
  res,
}: GetAndExtendCookieOnServerOpts): string | undefined {
  for (const cookieName of [name, ...migrationNames]) {
    const value = getCookieValueOnServer({ name: cookieName, req });
    if (value) {
      setCookieValueOnServer({ name, opts: { maxAge: extension }, res, value });
      // remove migrationName cookie because we've replaced it with the new name
      if (cookieName !== name) {
        clearCookieValueOnServer({ name: cookieName, res });
      }
      return value;
    }
  }
}

/**
 * Identical to `getAndExtendCookieValueOnServer` but is used when an object was stored instead of a string. The object is deserialized.
 */
export function getAndExtendCookieComplexValueOnServer<
  Payload extends VersionedPayload,
>(opts: GetAndExtendCookieOnServerOpts): Payload | undefined {
  return getComplexValue(getAndExtendCookieValueOnServer(opts));
}

export interface SetCookieOnServerOpts extends SetCookieValueOpts {
  res: Response;
}

/**
 * Express doesn't follow the cookie spec for max age and uses ms instead of seconds.
 */
export const generateCookieOptsExpress: typeof generateCookieOpts = (opts) => {
  const { maxAge, ...rest } = generateCookieOpts(opts);
  return {
    maxAge: maxAge !== undefined ? maxAge * 1000 : undefined,
    ...rest,
  };
};

/**
 * Properly sets a cookie value on a Response object serverside, adding default
 * configuration options.
 *
 * When storing a value other than a string (such as JSON / Objects) use `setCookieComplexValueOnServer`.
 */
export function setCookieValueOnServer({
  name,
  opts,
  res,
  value,
}: SetCookieOnServerOpts): void {
  const options = generateCookieOptsExpress({ domain: cookieDomain, ...opts });
  res.cookie(name, value, options);

  if (options.sameSite === 'none') {
    const { sameSite, ...optionsCompat } = options;
    res.cookie(name + SAMESITE_COMPAT, value, optionsCompat);
  }
}

interface SetCookieComplexOnServerOpts extends SetCookieValueComplexOpts {
  res: Response;
}

/**
 * Identical to `setCookieValueOnServer` but is used when storing an object instead of a string. The object is serialized and can be retrieved via `getAndExtendCookieComplexValueOnServer` or `getCookieComplexValueOnServer`.
 */
export function setCookieComplexValueOnServer(
  opts: SetCookieComplexOnServerOpts,
): void {
  return setCookieValueOnServer({
    ...opts,
    value: setComplexValue(opts.value),
  });
}

export interface ClearCookieOnServerOpts extends ClearCookieValueOpts {
  res: Response;
}

/**
 * Clears a cookie by name. In addition to name, takes an opts object for
 * configuring the various cookie options; path will default to '/' and domain
 * will default to the current default cookie domain.
 */
export function clearCookieValueOnServer({
  name,
  opts = {},
  res,
}: ClearCookieOnServerOpts): void {
  setCookieValueOnServer({
    name,
    opts: {
      ...opts,
      expires: new Date(0),
      maxAge: 0,
    },
    res,
    value: '',
  });
}
