import type { Request, Response } from 'express';
import { datatype, internet, random } from 'faker';
import {
  DEFAULT_COOKIE_DOMAIN,
  MAX_COOKIE_AGE,
  SAMESITE_COMPAT,
  setComplexValue,
} from 'tachyon-utils-twitch';
import {
  clearCookieValueOnServer,
  generateCookieOptsExpress,
  getAndExtendCookieComplexValueOnServer,
  getAndExtendCookieValueOnServer,
  getCookieComplexValueOnServer,
  getCookieValueOnServer,
  resetCookieDomainOnServer,
  setCookieComplexValueOnServer,
  setCookieDomainOnServer,
  setCookieValueOnServer,
} from '.';

describe('cookies', () => {
  afterAll(() => {
    resetCookieDomainOnServer();
  });

  beforeEach(() => {
    resetCookieDomainOnServer();
  });

  describe(getCookieValueOnServer, () => {
    it('returns a value when present in request', () => {
      const name = random.alphaNumeric();
      const value = random.alphaNumeric();
      const req = {
        cookies: {
          [name]: value,
        },
      } as Request;

      expect(getCookieValueOnServer({ name, req })).toEqual(value);
    });

    it('fallsback to sameSite compatability value when present in request', () => {
      const name = random.alphaNumeric();
      const value = random.alphaNumeric();
      const req = {
        cookies: {
          [name + SAMESITE_COMPAT]: value,
        },
      } as Request;

      expect(getCookieValueOnServer({ name, req })).toEqual(value);
    });

    it('returns undefined when no matching cookie in request', () => {
      const req = { cookies: {} } as Request;

      expect(
        getCookieValueOnServer({ name: random.alphaNumeric(), req }),
      ).toBeUndefined();
    });
  });

  describe(getCookieComplexValueOnServer, () => {
    it('returns a deserialized value when present in request', () => {
      const name = random.alphaNumeric();
      const complex = { foo: 'bar', version: 1 };
      const req = {
        cookies: {
          [name]: setComplexValue(complex),
        },
      } as Request;

      expect(getCookieComplexValueOnServer({ name, req })).toEqual(complex);
    });
  });

  describe(getAndExtendCookieValueOnServer, () => {
    it('extends the cookie value when present', () => {
      const name = random.alphaNumeric();
      const value = random.alphaNumeric();
      const req = {
        cookies: { [name]: value },
      } as Request;
      const res = { cookie: jest.fn() } as unknown as Response;

      expect(getAndExtendCookieValueOnServer({ name, req, res })).toEqual(
        value,
      );
      expect(res.cookie).toHaveBeenCalledWith(
        name,
        value,
        expect.objectContaining({ maxAge: MAX_COOKIE_AGE * 1000 }),
      );
    });

    it('does not extend the cookie value when missing', () => {
      const req = { cookies: {} } as Request;
      const res = { cookie: jest.fn() } as unknown as Response;

      expect(
        getAndExtendCookieValueOnServer({
          name: random.alphaNumeric(),
          req,
          res,
        }),
      ).toBeUndefined();
      expect(res.cookie).not.toHaveBeenCalled();
    });

    it('replaces the migrationName cookie with the new name', () => {
      const oldName = random.alphaNumeric();
      const value = random.alphaNumeric();
      const name = `tachyon-${oldName}`;
      const req = { cookies: { [oldName]: value } } as Request;
      const res = { cookie: jest.fn() } as unknown as Response;

      expect(
        getAndExtendCookieValueOnServer({
          migrationNames: [oldName],
          name,
          req,
          res,
        }),
      ).toEqual(value);
      expect(res.cookie).toHaveBeenCalledWith(
        name,
        value,
        expect.objectContaining({ maxAge: MAX_COOKIE_AGE * 1000 }),
      );
      expect(res.cookie).toHaveBeenCalledWith(
        oldName,
        '',
        expect.objectContaining({
          expires: new Date(0),
          maxAge: 0,
        }),
      );
    });

    it('fallsback to migrationNames', () => {
      const name = random.alphaNumeric();
      const value = random.alphaNumeric();
      const namespaced = `tachyon-${name}`;
      const req = { cookies: { [name]: value } } as Request;
      const res = { cookie: jest.fn() } as unknown as Response;

      expect(
        getAndExtendCookieValueOnServer({
          migrationNames: [name],
          name: namespaced,
          req,
          res,
        }),
      ).toEqual(value);
    });

    it('uses name over migrationNames when name is present', () => {
      const name = random.alphaNumeric();
      const value = random.alphaNumeric();
      const namespaced = `tachyon-${name}`;
      const req = {
        cookies: { [name]: value, [namespaced]: 'foo' },
      } as Request;
      const res = { cookie: jest.fn() } as unknown as Response;

      expect(
        getAndExtendCookieValueOnServer({
          migrationNames: [name],
          name: namespaced,
          req,
          res,
        }),
      ).toEqual('foo');
    });
  });

  describe(getAndExtendCookieComplexValueOnServer, () => {
    it('returns a deserialized value when present in request', () => {
      const name = random.alphaNumeric();
      const complex = { foo: 'bar', version: 1 };
      const req = {
        cookies: {
          [name]: setComplexValue(complex),
        },
      } as Request;
      const res = { cookie: jest.fn() } as unknown as Response;

      expect(
        getAndExtendCookieComplexValueOnServer({ name, req, res }),
      ).toEqual(complex);
    });
  });

  describe(setCookieValueOnServer, () => {
    it('adds cookie to response', () => {
      const name = random.alphaNumeric();
      const value = random.alphaNumeric();
      const opts = { maxAge: datatype.number() };
      const res = { cookie: jest.fn() } as any as Response;

      setCookieValueOnServer({ name, opts, res, value });
      expect(res.cookie).toHaveBeenCalledWith(
        name,
        value,
        generateCookieOptsExpress(opts),
      );
    });

    it('adds sameSite compatability cookie to response when sameSite is set to none', () => {
      const name = random.alphaNumeric();
      const value = random.alphaNumeric();
      const opts = { maxAge: datatype.number() };
      const res = { cookie: jest.fn() } as any as Response;
      const options = generateCookieOptsExpress(opts);
      delete options.sameSite;

      setCookieValueOnServer({ name, opts, res, value });
      expect(res.cookie).toHaveBeenCalledWith(
        name + SAMESITE_COMPAT,
        value,
        options,
      );
    });
  });

  describe(setCookieComplexValueOnServer, () => {
    it('sets the serialized value', () => {
      const name = random.alphaNumeric();
      const value = { foo: 'bar', version: 1 };
      const opts = { maxAge: datatype.number() };
      const res = { cookie: jest.fn() } as any as Response;

      setCookieComplexValueOnServer({ name, opts, res, value });
      expect(res.cookie).toHaveBeenCalledWith(
        name,
        setComplexValue(value),
        generateCookieOptsExpress(opts),
      );
    });
  });

  describe(setCookieDomainOnServer, () => {
    it('properly works with a custom domain', () => {
      const name = random.alphaNumeric();
      const value = random.alphaNumeric();
      const opts = { maxAge: datatype.number() };
      const res = { cookie: jest.fn() } as any as Response;
      const domain = internet.url();

      setCookieDomainOnServer(domain);
      setCookieValueOnServer({ name, opts, res, value });
      expect(res.cookie).toHaveBeenCalledWith(
        name,
        value,
        expect.objectContaining({
          domain,
        }),
      );
    });
  });

  describe(clearCookieValueOnServer, () => {
    it('removes a cookie with default opts', () => {
      const name = random.alphaNumeric();
      const res = { cookie: jest.fn() } as any as Response;

      clearCookieValueOnServer({ name, res });

      expect(res.cookie).toHaveBeenCalledWith(
        name,
        '',
        expect.objectContaining({
          domain: DEFAULT_COOKIE_DOMAIN,
          expires: new Date(0),
          maxAge: 0,
        }),
      );
    });

    it('removes the sameSite compatability cookie', () => {
      const name = random.alphaNumeric();
      const res = { cookie: jest.fn() } as any as Response;

      clearCookieValueOnServer({ name, res });

      expect(res.cookie).toHaveBeenCalledWith(
        name + SAMESITE_COMPAT,
        '',
        expect.objectContaining({
          domain: DEFAULT_COOKIE_DOMAIN,
          expires: new Date(0),
          maxAge: 0,
        }),
      );
    });
  });
});
