import type { ExperimentBucket } from 'tachyon-experiments';
import { parseBucket } from 'tachyon-experiments';
import { logger } from 'tachyon-logger';
import type { TachyonRequest } from 'tachyon-next-types';
import { flattenHeaderOrParam } from 'tachyon-utils-stdlib';

type Opts = {
  header: string;
  package: string;
};

/**
 * Generates an experiment bucket from a request object (based on headers).
 * See "Experiment Buckets" in the docs for more info:
 * https://git.xarth.tv/pages/emerging-platforms/tachyon/d/apps/tomorrow/processes/managing-experiments/
 *
 * @return bucket, or the undefined if the experiment bucket header is not present
 */
export function getExperimentBucket(
  req: TachyonRequest,
  opts: Opts,
): ExperimentBucket | undefined {
  const experimentBucket = req.headers[opts.header];

  if (Array.isArray(experimentBucket) && experimentBucket.length > 1) {
    reportTooManyBuckets(experimentBucket.join(', '), opts.package);
  }

  if (typeof experimentBucket === 'string' && /,/.test(experimentBucket)) {
    reportTooManyBuckets(experimentBucket, opts.package);
  }

  return parseBucket(flattenHeaderOrParam(experimentBucket));
}

function reportTooManyBuckets(buckets: string, pkg: string): void {
  logger.error({
    category: 'getExperimentBucket',
    context: {
      buckets,
    },
    message: 'bucket_error: too many buckets detected',
    package: pkg,
  });
}
