import { logger } from 'tachyon-logger';
import type { TachyonRequest } from 'tachyon-next-types';
import { getExperimentBucket } from '.';

jest.mock('tachyon-logger', () => ({ logger: { error: jest.fn() } }));

const EXPERIMENT_BUCKET_HEADER = 'x-bucket-id';

describe(getExperimentBucket, () => {
  beforeEach(() => {
    (logger.error as jest.Mock).mockReset();
  });

  it('returns undefined when there is no header data', () => {
    expect(
      getExperimentBucket(
        {
          headers: {},
        } as any as TachyonRequest,
        { header: EXPERIMENT_BUCKET_HEADER, package: 'server-utils' },
      ),
    ).toBeUndefined();
  });

  it('returns undefined when there is invalid header data', () => {
    expect(
      getExperimentBucket(
        {
          headers: {
            [EXPERIMENT_BUCKET_HEADER]: 'q',
          },
        } as any as TachyonRequest,
        { header: EXPERIMENT_BUCKET_HEADER, package: 'server-utils' },
      ),
    ).toBeUndefined();
  });

  it('parses the bucketID into the bucket value', () => {
    expect(
      getExperimentBucket(
        {
          headers: {
            [EXPERIMENT_BUCKET_HEADER]: 'f',
          },
        } as any as TachyonRequest,
        { header: EXPERIMENT_BUCKET_HEADER, package: 'server-utils' },
      ),
    ).toEqual('f');
  });

  it('uses a first element of string[] bucketID as bucket value', () => {
    expect(
      getExperimentBucket(
        {
          headers: {
            [EXPERIMENT_BUCKET_HEADER]: ['f', 'a'],
          },
        } as any as TachyonRequest,
        { header: EXPERIMENT_BUCKET_HEADER, package: 'server-utils' },
      ),
    ).toEqual('f');
  });

  it('reports when we hit too many buckets as a string', () => {
    getExperimentBucket(
      {
        headers: {
          [EXPERIMENT_BUCKET_HEADER]: 'a,f',
        },
      } as any as TachyonRequest,
      { header: EXPERIMENT_BUCKET_HEADER, package: 'server-utils' },
    );

    expect(logger.error).toHaveBeenCalledWith({
      category: 'getExperimentBucket',
      context: {
        buckets: 'a,f',
      },
      message: 'bucket_error: too many buckets detected',
      package: 'server-utils',
    });
  });

  it('reports when we hit too many buckets as an array', () => {
    getExperimentBucket(
      {
        headers: {
          [EXPERIMENT_BUCKET_HEADER]: ['a', 'f'],
        },
      } as any as TachyonRequest,
      { header: EXPERIMENT_BUCKET_HEADER, package: 'server-utils' },
    );

    expect(logger.error).toHaveBeenCalledWith({
      category: 'getExperimentBucket',
      context: {
        buckets: 'a, f',
      },
      message: 'bucket_error: too many buckets detected',
      package: 'server-utils',
    });
  });
});
