import type { Request } from 'express';
import { logger } from 'tachyon-logger';
import { RedirectionCode } from 'tachyon-type-library';
import type {
  RedirectCommand,
  RedirectManager,
} from '../redirectManagerExpressAdapter';

export interface BadRequestMiddlewareOpts {
  badRequestRedirectPath: string;
  dynamicPathnames: string[];
}

/**
 * Checks for known bad request paths and redirects them to our app error
 * route to avoid an ungraceful server error.
 */
export function badRequestManager({
  badRequestRedirectPath,
  dynamicPathnames,
}: BadRequestMiddlewareOpts): RedirectManager {
  return (request: Request): RedirectCommand => {
    function isBadRequest(): boolean {
      const { path: requestPath } = request;

      // express will attempt this call, so we try it first to prevent errors
      try {
        decodeURI(requestPath);
      } catch {
        return true;
      }

      // prevent unsubstituted versions of dynamic paths from breaking next
      // e.g. /[login]
      // Necessary until https://github.com/vercel/next.js/issues/17096 is fixed
      if (dynamicPathnames.includes(requestPath)) {
        // we log this specific case to track frequency
        logger.log({
          category: 'badRequestMiddleware',
          context: {
            referer: request.header('Referer') || '',
            userAgent: request.header('User-Agent') || '',
          },
          message: `Dynamic route pathname requested: ${requestPath}`,
          package: 'tachyon-server-utils',
        });
        return true;
      }

      return false;
    }

    if (isBadRequest()) {
      return {
        location: badRequestRedirectPath,
        statusCode: RedirectionCode.TemporaryRedirect,
      };
    }
  };
}
